#!/bin/tcsh
# makejulich_afni
# script to convert downloaded Julich atlases from EBRAINS website
# into datasets that are useful in AFNI

set orig_dir =  maximum-probability-maps_MPMs_207-areas
set lh_MNI_orig = $orig_dir/JulichBrainAtlas_3.1_207areas_MPM_lh_MNI152.nii.gz
set rh_MNI_orig = $orig_dir/JulichBrainAtlas_3.1_207areas_MPM_rh_MNI152.nii.gz

set lh_N27_orig = $orig_dir/JulichBrainAtlas_3.1_207areas_MPM_lh_Colin27.nii.gz
set rh_N27_orig = $orig_dir/JulichBrainAtlas_3.1_207areas_MPM_rh_Colin27.nii.gz

# labels are distributed in xml files for each hemisphere,
# but both are identical
set labels = ( \
   $orig_dir/JulichBrainAtlas_3.1_207areas_MPM_lh_MNI152.xml \
   $orig_dir/JulichBrainAtlas_3.1_207areas_MPM_rh_MNI152.xml \
)

# refit space to MNI2009c and make viewable with integer colormaps

# pull indices and labels out of xml file
# example:   206  Area 7PC (SPL)  
grep "grayvalue=" $labels[1] | \
   awk -F = '{print $4}'|sed 's/[\"\>\<]/ /g'|sed 's/\/Structure//g' \
      > lh_index_labels.txt

# sort by index
sort -n -k 1 lh_index_labels.txt > lh_index_labels_sort.txt

rm ?h_index_labels_AFNI.txt

set nl = `wc -l lh_index_labels_sort.txt`
foreach ii (`count_afni -digits 3 1 $nl[1]` )
   # get line from sorted labels file
   set ill = `head -$ii lh_index_labels_sort.txt |tail -1`

   # get index
   set ind = $ill[1]
   # 3 digit form was complicating @MakeLabelTable because of octal conversion
   # 008, 009 not interpretable in octal... so removing this here
   #set ind = `ccalc -form "%.3d" -expr "$ind"`
   # version for right hemisphere - add 300 to make different 
   set rind = `ccalc -form "%.3d" -expr "$ind+300"`

   # distinguish label and longname for each region
   # replace spaces with underscores in main label
   set lbl = `echo $ill[2-] | awk -F\( '{print $1}' | \
              sed 's/\ $//g'| sed 's/ /_/g'`   

   set lnm = `echo $ill[2-] | awk -F\( '{print $2}' | \
              sed 's/)//g'`

   echo "$ind left_$lbl $lnm" >> lh_index_labels_AFNI.txt
   echo "$rind right_$lbl $lnm" >> rh_index_labels_AFNI.txt

end

cat lh_index_labels_AFNI.txt rh_index_labels_AFNI.txt > lr_index_labels_AFNI.txt

# add 300 to right side and combine left-right hemispheres
# data was stored efficiently as byte for data 1-207, but
# adding 300 goes beyond 255, so need short integers 
# MNI 2009c version
3dcalc -a $lh_MNI_orig -b $rh_MNI_orig -expr 'a*not(b)+(b+300)*step(b)' \
   -datum short -prefix Julich_MNI2009c_v3.1.nii.gz -overwrite
# N27 version (there is a bit of overlap in left-right, but given priority 
# to right side)
3dcalc -a $lh_N27_orig -b $rh_N27_orig -expr 'a*not(b)+(b+300)*step(b)' \
   -datum short -prefix Julich_MNI_N27_v3.1.nii.gz -overwrite

# make into atlas for MNI 2009c space
@Atlasize -dset Julich_MNI2009c_v3.1.nii.gz -space MNI_2009c_asym \
   -lab_file lr_index_labels_AFNI.txt 1 0 \
   -longnames 2 \
   -atlas_type S \
   -atlas_description "JulichBrain v3.1 for MNI_2009c" \
   -atlas_name "Julich_MNI2009c_v3.1" \
   -centers -centertype Icent \
   -skip_novoxels

# make into atlas for MNI N27 space (same labels apply to both spaces)
@Atlasize -dset Julich_MNI_N27_v3.1.nii.gz -space MNI_N27 \
   -lab_file lr_index_labels_AFNI.txt 1 0 \
   -longnames 2 \
   -atlas_type S \
   -atlas_description "JulichBrain v3.1 for MNI_N27" \
   -atlas_name "Julich_MNI_N27_v3.1" \
   -centers -centertype Icent \
   -skip_novoxels


# set version information in AFNI header extension 
# and NIFTI format description field too
set atlas = Julich_MNI2009c_v3.1.nii.gz
3drefit -sublabel 0 REGIONS  $atlas
3drefit -space MNI_2009c_asym $atlas
3drefit -denote $atlas
3dNotes -a ATLASVER:3.1 $atlas
3dNotes -a "ATLASDESC:JulichBrain MNI2009c_asym" $atlas
gunzip $atlas
nifti_tool -mod_field descrip "ATLASVER:3.1 ATLASDESC:JulichBrain MNI2009c_asym" \
   -infiles Julich_MNI2009c_v3.1.nii -mod_hdr -overwrite
gzip Julich_MNI2009c_v3.1.nii

set atlas = Julich_MNI_N27_v3.1.nii.gz
3drefit -sublabel 0 REGIONS  $atlas
3drefit -space MNI_N27 $atlas
3drefit -denote $atlas
3dNotes -a ATLASVER:3.1 $atlas
3dNotes -a "ATLASDESC:JulichBrain MNI N27" $atlas
gunzip $atlas
nifti_tool -mod_field descrip "ATLASVER:3.0.3 ATLASDESC:JulichBrain MNI N27" \
   -infiles Julich_MNI_N27_v3.1.nii -mod_hdr -overwrite
gzip Julich_MNI_N27_v3.1.nii


# update @MakeLabelTable with modified version for 008 problems
