#ifndef __PBARDEFS_INCLUDED__
#define __PBARDEFS_INCLUDED__

/*******************************************************/
/* There are 2 simple ways to make a colormap,
   both of which are used in the header.
   Look in pbar.c to see how the colormaps are
   created in the AFNI GUI.

    * N byte triples -> PBAR_define_bigmap()

      Here, you define a char array, one giant
      string, using spaces to delimit the N+1
      sub-strings:
        Name #rrggbb #rrggbb ... (N of these)
      where the N #rrggbb values each have six
      hex digits defining the colors. In this
      usage, the number N of defining colors
      is inferred from the number of sub-strings
      in the big input string.
      The CytoArch_ROI_i256 map is a sample.

    * N float triples -> PBAR_define_bigmap_float()

     ** All these declarations are now in file **
     ** pbardefs_float.h [24 Oct 2019 - RWCox] **

      Here, you define a float array, length 3*N,
      which for each color contains the 3 RGB values
      (range = 0..1), in the order
        R0 G0 B0 R1 G1 B1 R2 G2 B2 ...
      In this usage, the number N of defining colors
      must be supplied explicitly, as is the colormap
      name to be used in menus.
      The googleturbo map is a sample; e.g., the
      last color triple there is
         0.47960, 0.01583, 0.01055
      which is mostly Red. The number N and the name
      are provided by
        #define      googleturbo_num 256
        static char *googleturbo_name = "GoogleTurbo" ;
      which are passed to PBAR_define_bigmap_float().

   Internally, a colormap is 256 colors long
   (unless option '-XXXnpane' is provided on the AFNI
   GUI command line.). Providing fewer colors will mean
   that the extra colors are provided by interpolation
   in the array of RGB values provided.

   There are more complicated ways to make a colormap.
   See pbar.c (at the risk of your sanity).
*//*****************************************************/

/* ZSS: New 256-color maps.
   Each new color map is added with
   a call to PBAR_define_bigmap in pbar.c
*/

/* Color map for The Eickhoff and Zilles
   Cytoarchitectonic data AllAreas_v12

  see matlab/MakeCytoArchColMaps.m
*/
static char CYTOARCH_ROI_256_CMD[] = {
      "CytoArch_ROI_i256 "
      "#84392e #018f67 #776b1b #01d9d4 #9f35b8 "
      "#f7d202 #2100cf #d494f6 #bbfa2d #fd2d74 "
      "#90c648 #c4516e #fe28fd #a83d26 #4fda69 "
      "#40fcf8 #c4589f #27abf3 #610051 #49f8c0 "
      "#ad185d #68f88d #d80491 #d92918 #2b649d "
      "#65bbea #772e83 #fed039 #0591b2 #ed2b3e "
      "#67a84b #cb7941 #7a2eb0 #39af94 #57fa28 "
      "#fb7dfd #a46b42 #13e94c #4b70f0 #c8cf33 "
      "#02e79a #fcae11 #96d7fb #42358b #0009fd "
      "#ff2bcf #9d9909 #173472 #fe7c31 #3780c8 "
      "#1041fe #f0d260 #a19349 #45ff8a #fd9376 "
      "#cf3fcc #c3c866 #059489 #3cfe59 #ff0091 "
      "#fbb3a1 #c8f104 #d92a70 #86ef42 #7b02ea "
      "#fde693 #b408b2 #1b52d4 #f687cd #8257b9 "
      "#88100b #fa03cb #e85783 #045602 #7d97fa "
      "#8cfe76 #1608a9 #fdff01 #ff56b9 #0d359a "
      "#b41335 #7b07ab #3dc0c5 #feab3d #6fc9af "
      "#0afdf9 #b9f760 #4cc50f #c067fe #fffb5f "
      "#df7e05 #ba00de #3d954b #c636f6 #05b7c2 "
      "#ac3d02 #0d7e3d #534fb4 #ac7119 #ab0a85 "
      "#0667a5 #17b80f #494be9 #860b3b #85c323 "
      "#6ddedd #3d888c #d5956d #378307 #390872 "
      "#8463ff #24f0c6 #d1b100 #02fc73 #90ffa8 "
      "#01c6fc #fd1017 #4407f2 #9b4450 #03b24b "
      "#a0428e #02bf89 #4d96ff #79490c #461ab6 "
      "#43d344 #b06dc9 #df4e04 #83d16f #04f025 "
      "#a2fee4 #e80153 #ffa6f8 #96f718 #5e83c0 "
      "#eb02f9 #bf0005 #cfa736 #0a7fdf #f685a0 "
      "#000a7a #e12ba0 #046372 #d1eb91 #740377 "
      "#39d6fc #708e1f #acb0fd #fd52fa #01840a "
      "#0626d6 #5c0a08 #e3f92f #49ff00 #3ed78e "
      "#872cde #df5146 #b3b3b3 #b1b1b1 #afafaf "
      "#aeaeae #acacac #ababab #a9a9a9 #a8a8a8 "
      "#a6a6a6 #a4a4a4 #a3a3a3 #a1a1a1 #a0a0a0 "
      "#9e9e9e #9d9d9d #9b9b9b #9a9a9a #989898 "
      "#969696 #959595 #939393 #929292 #909090 "
      "#8f8f8f #8d8d8d #8b8b8b #8a8a8a #888888 "
      "#878787 #858585 #848484 #828282 #818181 "
      "#7f7f7f #7d7d7d #7c7c7c #7a7a7a #797979 "
      "#777777 #767676 #747474 #727272 #717171 "
      "#6f6f6f #6e6e6e #6c6c6c #6b6b6b #696969 "
      "#686868 #666666 #646464 #636363 #616161 "
      "#606060 #5e5e5e #5d5d5d #5b5b5b #5a5a5a "
      "#585858 #565656 #555555 #535353 #525252 "
      "#505050 #4f4f4f #4d4d4d #4b4b4b #4a4a4a "
      "#484848 #474747 #454545 #444444 #424242 "
      "#414141 #3f3f3f #3d3d3d #3c3c3c #3a3a3a "
      "#393939 #373737 #363636 #343434 #323232 "
      "#313131 #2f2f2f #2e2e2e #2c2c2c #2b2b2b "
      "#292929 #282828 #262626 #242424 #232323 "
      "#212121 #202020 #1e1e1e #1d1d1d #1b1b1b "
      "#1a1a1a "
};
static char CYTOARCH_ROI_256_GAP_CMD[] = {
      "CytoArch_ROI_i256_gap "
      "#84392e #018f67 #776b1b #01d9d4 #9f35b8 "
      "#f7d202 #2100cf #d494f6 #bbfa2d #fd2d74 "
      "#90c648 #c4516e #fe28fd #a83d26 #4fda69 "
      "#40fcf8 #c4589f #27abf3 #610051 #49f8c0 "
      "#ad185d #68f88d #d80491 #d92918 #2b649d "
      "#65bbea #772e83 #fed039 #0591b2 #ed2b3e "
      "#67a84b #cb7941 #7a2eb0 #39af94 #57fa28 "
      "#fb7dfd #a46b42 #13e94c #4b70f0 #c8cf33 "
      "#02e79a #fcae11 #96d7fb #42358b #0009fd "
      "#ff2bcf #9d9909 #173472 #fe7c31 #3780c8 "
      "#1041fe #f0d260 #a19349 #45ff8a #fd9376 "
      "#cf3fcc #c3c866 #059489 #3cfe59 #ff0091 "
      "#fbb3a1 #c8f104 #d92a70 #86ef42 #7b02ea "
      "#fde693 #b408b2 #1b52d4 #f687cd #8257b9 "
      "#88100b #fa03cb #e85783 #045602 #7d97fa "
      "#8cfe76 #1608a9 #fdff01 #ff56b9 #0d359a "
      "#b41335 #7b07ab #3dc0c5 #feab3d #6fc9af "
      "#0afdf9 #b9f760 #4cc50f #c067fe #fffb5f "
      "#df7e05 #ba00de #3d954b #c636f6 #05b7c2 "
      "#ac3d02 #0d7e3d #534fb4 #ac7119 #ab0a85 "
      "#0667a5 #17b80f #494be9 #860b3b #85c323 "
      "#6ddedd #3d888c #d5956d #378307 #390872 "
      "#8463ff #24f0c6 #d1b100 #02fc73 #90ffa8 "
      "#01c6fc #fd1017 #4407f2 #9b4450 #03b24b "
      "#a0428e #02bf89 #4d96ff #79490c #461ab6 "
      "#43d344 #b06dc9 #df4e04 #83d16f #04f025 "
      "#a2fee4 #e80153 #ffa6f8 #96f718 #5e83c0 "
      "#eb02f9 #bf0005 #cfa736 #0a7fdf #f685a0 "
      "#000a7a #e12ba0 #046372 #d1eb91 #740377 "
      "#39d6fc #708e1f #acb0fd #fd52fa #01840a "
      "#0626d6 #5c0a08 #e3f92f #49ff00 #3ed78e "
      "#872cde #df5146 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 #000000 #000000 #000000 #000000 "
      "#000000 "
};
/* While we're at it, some maps for ROI drawing
   see matlab/MakeROIColMaps.m
*/
static char ROI_256_CMD[] = {
      "ROI_i256 "
      "#41a2b0 #0a32b6 #df70f8 #78f525 #d80475 "
      "#cf7c23 #fbb060 #5df345 #fc3820 #d97d98 "
      "#15b76c #fd8b14 #83b6ed #a544fa #adbd02 "
      "#8f0459 #90e18d #d90ad8 #56146f #59dffb "
      "#3f07d3 #fe2e95 #ac1f0a #01c12c #8ee1d6 "
      "#6bfa67 #2adb1f #4e29fb #dd0203 #086420 "
      "#de53fc #47b5f6 #7f2261 #7e9601 #a748db "
      "#21dfa7 #68a8c5 #02f804 #d05024 #d8fd74 "
      "#0007ce #fb7576 #07d66a #69022b #02f8cd "
      "#650acc #f80372 #be3b45 #fdb37f #3ea0cd "
      "#ff5a57 #e0f450 #a69229 #2504fb #6ecf91 "
      "#8a04cc #2d7dfd #22ffa1 #0660f7 #256401 "
      "#450195 #c9d009 #2b822b #7a33fd #c77461 "
      "#25b548 #adec88 #c235a1 #67f4ab #d305b5 "
      "#4ba51a #9f1cff #60b56c #045650 #ef60da "
      "#622ad6 #6efee7 #c101ff #fd8654 #a6c21f "
      "#28d66d #ff5c04 #3029da #82cc03 #8c89d9 "
      "#f35624 #65cf44 #090355 #b18ff1 #dc0327 "
      "#8564dd #20ef02 #04a4f0 #515e02 #3041ad "
      "#30fb2a #d283ce #6df405 #3cad6e #84392e "
      "#018f67 #776b1b #01d9d4 #9f35b8 #f7d202 "
      "#2100cf #d494f6 #bbfa2d #fd2d74 #90c648 "
      "#c4516e #fe28fd #a83d26 #4fda69 #40fcf8 "
      "#c4589f #27abf3 #610051 #49f8c0 #ad185d "
      "#68f88d #d80491 #d92918 #2b649d #65bbea "
      "#772e83 #fed039 #0591b2 #ed2b3e #67a84b "
      "#cb7941 #7a2eb0 #39af94 #57fa28 #fb7dfd "
      "#a46b42 #13e94c #4b70f0 #c8cf33 #02e79a "
      "#fcae11 #96d7fb #42358b #0009fd #ff2bcf "
      "#9d9909 #173472 #fe7c31 #3780c8 #1041fe "
      "#f0d260 #a19349 #45ff8a #fd9376 #cf3fcc "
      "#c3c866 #059489 #3cfe59 #ff0091 #fbb3a1 "
      "#c8f104 #d92a70 #86ef42 #7b02ea #fde693 "
      "#b408b2 #1b52d4 #f687cd #8257b9 #88100b "
      "#fa03cb #e85783 #045602 #7d97fa #8cfe76 "
      "#1608a9 #fdff01 #ff56b9 #0d359a #b41335 "
      "#7b07ab #3dc0c5 #feab3d #6fc9af #0afdf9 "
      "#b9f760 #4cc50f #c067fe #fffb5f #df7e05 "
      "#ba00de #3d954b #c636f6 #05b7c2 #ac3d02 "
      "#0d7e3d #534fb4 #ac7119 #ab0a85 #0667a5 "
      "#17b80f #494be9 #860b3b #85c323 #6ddedd "
      "#3d888c #d5956d #378307 #390872 #8463ff "
      "#24f0c6 #d1b100 #02fc73 #90ffa8 #01c6fc "
      "#fd1017 #4407f2 #9b4450 #03b24b #a0428e "
      "#02bf89 #4d96ff #79490c #461ab6 #43d344 "
      "#b06dc9 #df4e04 #83d16f #04f025 #a2fee4 "
      "#e80153 #ffa6f8 #96f718 #5e83c0 #eb02f9 "
      "#bf0005 #cfa736 #0a7fdf #f685a0 #000a7a "
      "#e12ba0 #046372 #d1eb91 #740377 #39d6fc "
      "#708e1f #acb0fd #fd52fa #01840a #0626d6 "
      "#5c0a08 #e3f92f #49ff00 #3ed78e #872cde "
      "#df5146 "
};

static char ROI_128_CMD[] = {
      "ROI_i128 "
      "#ed2b3e #67a84b #cb7941 #7a2eb0 #39af94 "
      "#57fa28 #fb7dfd #a46b42 #13e94c #4b70f0 "
      "#c8cf33 #02e79a #fcae11 #96d7fb #42358b "
      "#0009fd #ff2bcf #9d9909 #173472 #fe7c31 "
      "#3780c8 #1041fe #f0d260 #a19349 #45ff8a "
      "#fd9376 #cf3fcc #c3c866 #059489 #3cfe59 "
      "#ff0091 #fbb3a1 #c8f104 #d92a70 #86ef42 "
      "#7b02ea #fde693 #b408b2 #1b52d4 #f687cd "
      "#8257b9 #88100b #fa03cb #e85783 #045602 "
      "#7d97fa #8cfe76 #1608a9 #fdff01 #ff56b9 "
      "#0d359a #b41335 #7b07ab #3dc0c5 #feab3d "
      "#6fc9af #0afdf9 #b9f760 #4cc50f #c067fe "
      "#fffb5f #df7e05 #ba00de #3d954b #c636f6 "
      "#05b7c2 #ac3d02 #0d7e3d #534fb4 #ac7119 "
      "#ab0a85 #0667a5 #17b80f #494be9 #860b3b "
      "#85c323 #6ddedd #3d888c #d5956d #378307 "
      "#390872 #8463ff #24f0c6 #d1b100 #02fc73 "
      "#90ffa8 #01c6fc #fd1017 #4407f2 #9b4450 "
      "#03b24b #a0428e #02bf89 #4d96ff #79490c "
      "#461ab6 #43d344 #b06dc9 #df4e04 #83d16f "
      "#04f025 #a2fee4 #e80153 #ffa6f8 #96f718 "
      "#5e83c0 #eb02f9 #bf0005 #cfa736 #0a7fdf "
      "#f685a0 #000a7a #e12ba0 #046372 #d1eb91 "
      "#740377 #39d6fc #708e1f #acb0fd #fd52fa "
      "#01840a #0626d6 #5c0a08 #e3f92f #49ff00 "
      "#3ed78e #872cde #df5146 "
};
static char ROI_128_256_CMD[] = {
      "ROI_i128 "
      "#ed2b3e #ed2b3e #67a84b #67a84b #cb7941 "
      "#cb7941 #7a2eb0 #7a2eb0 #39af94 #39af94 "
      "#57fa28 #57fa28 #fb7dfd #fb7dfd #a46b42 "
      "#a46b42 #13e94c #13e94c #4b70f0 #4b70f0 "
      "#c8cf33 #c8cf33 #02e79a #02e79a #fcae11 "
      "#fcae11 #96d7fb #96d7fb #42358b #42358b "
      "#0009fd #0009fd #ff2bcf #ff2bcf #9d9909 "
      "#9d9909 #173472 #173472 #fe7c31 #fe7c31 "
      "#3780c8 #3780c8 #1041fe #1041fe #f0d260 "
      "#f0d260 #a19349 #a19349 #45ff8a #45ff8a "
      "#fd9376 #fd9376 #cf3fcc #cf3fcc #c3c866 "
      "#c3c866 #059489 #059489 #3cfe59 #3cfe59 "
      "#ff0091 #ff0091 #fbb3a1 #fbb3a1 #c8f104 "
      "#c8f104 #d92a70 #d92a70 #86ef42 #86ef42 "
      "#7b02ea #7b02ea #fde693 #fde693 #b408b2 "
      "#b408b2 #1b52d4 #1b52d4 #f687cd #f687cd "
      "#8257b9 #8257b9 #88100b #88100b #fa03cb "
      "#fa03cb #e85783 #e85783 #045602 #045602 "
      "#7d97fa #7d97fa #8cfe76 #8cfe76 #1608a9 "
      "#1608a9 #fdff01 #fdff01 #ff56b9 #ff56b9 "
      "#0d359a #0d359a #b41335 #b41335 #7b07ab "
      "#7b07ab #3dc0c5 #3dc0c5 #feab3d #feab3d "
      "#6fc9af #6fc9af #0afdf9 #0afdf9 #b9f760 "
      "#b9f760 #4cc50f #4cc50f #c067fe #c067fe "
      "#fffb5f #fffb5f #df7e05 #df7e05 #ba00de "
      "#ba00de #3d954b #3d954b #c636f6 #c636f6 "
      "#05b7c2 #05b7c2 #ac3d02 #ac3d02 #0d7e3d "
      "#0d7e3d #534fb4 #534fb4 #ac7119 #ac7119 "
      "#ab0a85 #ab0a85 #0667a5 #0667a5 #17b80f "
      "#17b80f #494be9 #494be9 #860b3b #860b3b "
      "#85c323 #85c323 #6ddedd #6ddedd #3d888c "
      "#3d888c #d5956d #d5956d #378307 #378307 "
      "#390872 #390872 #8463ff #8463ff #24f0c6 "
      "#24f0c6 #d1b100 #d1b100 #02fc73 #02fc73 "
      "#90ffa8 #90ffa8 #01c6fc #01c6fc #fd1017 "
      "#fd1017 #4407f2 #4407f2 #9b4450 #9b4450 "
      "#03b24b #03b24b #a0428e #a0428e #02bf89 "
      "#02bf89 #4d96ff #4d96ff #79490c #79490c "
      "#461ab6 #461ab6 #43d344 #43d344 #b06dc9 "
      "#b06dc9 #df4e04 #df4e04 #83d16f #83d16f "
      "#04f025 #04f025 #a2fee4 #a2fee4 #e80153 "
      "#e80153 #ffa6f8 #ffa6f8 #96f718 #96f718 "
      "#5e83c0 #5e83c0 #eb02f9 #eb02f9 #bf0005 "
      "#bf0005 #cfa736 #cfa736 #0a7fdf #0a7fdf "
      "#f685a0 #f685a0 #000a7a #000a7a #e12ba0 "
      "#e12ba0 #046372 #046372 #d1eb91 #d1eb91 "
      "#740377 #740377 #39d6fc #39d6fc #708e1f "
      "#708e1f #acb0fd #acb0fd #fd52fa #fd52fa "
      "#01840a #01840a #0626d6 #0626d6 #5c0a08 "
      "#5c0a08 #e3f92f #e3f92f #49ff00 #49ff00 "
      "#3ed78e #3ed78e #872cde #872cde #df5146 "
      "#df5146 "
};

static char ROI_64_CMD[] = {
      "ROI_i64 "
      "#c636f6 #05b7c2 #ac3d02 #0d7e3d #534fb4 "
      "#ac7119 #ab0a85 #0667a5 #17b80f #494be9 "
      "#860b3b #85c323 #6ddedd #3d888c #d5956d "
      "#378307 #390872 #8463ff #24f0c6 #d1b100 "
      "#02fc73 #90ffa8 #01c6fc #fd1017 #4407f2 "
      "#9b4450 #03b24b #a0428e #02bf89 #4d96ff "
      "#79490c #461ab6 #43d344 #b06dc9 #df4e04 "
      "#83d16f #04f025 #a2fee4 #e80153 #ffa6f8 "
      "#96f718 #5e83c0 #eb02f9 #bf0005 #cfa736 "
      "#0a7fdf #f685a0 #000a7a #e12ba0 #046372 "
      "#d1eb91 #740377 #39d6fc #708e1f #acb0fd "
      "#fd52fa #01840a #0626d6 #5c0a08 #e3f92f "
      "#49ff00 #3ed78e #872cde #df5146 "
};
static char ROI_64_256_CMD[] = {
      "ROI_i64 "
      "#c636f6 #c636f6 #c636f6 #c636f6 #05b7c2 "
      "#05b7c2 #05b7c2 #05b7c2 #ac3d02 #ac3d02 "
      "#ac3d02 #ac3d02 #0d7e3d #0d7e3d #0d7e3d "
      "#0d7e3d #534fb4 #534fb4 #534fb4 #534fb4 "
      "#ac7119 #ac7119 #ac7119 #ac7119 #ab0a85 "
      "#ab0a85 #ab0a85 #ab0a85 #0667a5 #0667a5 "
      "#0667a5 #0667a5 #17b80f #17b80f #17b80f "
      "#17b80f #494be9 #494be9 #494be9 #494be9 "
      "#860b3b #860b3b #860b3b #860b3b #85c323 "
      "#85c323 #85c323 #85c323 #6ddedd #6ddedd "
      "#6ddedd #6ddedd #3d888c #3d888c #3d888c "
      "#3d888c #d5956d #d5956d #d5956d #d5956d "
      "#378307 #378307 #378307 #378307 #390872 "
      "#390872 #390872 #390872 #8463ff #8463ff "
      "#8463ff #8463ff #24f0c6 #24f0c6 #24f0c6 "
      "#24f0c6 #d1b100 #d1b100 #d1b100 #d1b100 "
      "#02fc73 #02fc73 #02fc73 #02fc73 #90ffa8 "
      "#90ffa8 #90ffa8 #90ffa8 #01c6fc #01c6fc "
      "#01c6fc #01c6fc #fd1017 #fd1017 #fd1017 "
      "#fd1017 #4407f2 #4407f2 #4407f2 #4407f2 "
      "#9b4450 #9b4450 #9b4450 #9b4450 #03b24b "
      "#03b24b #03b24b #03b24b #a0428e #a0428e "
      "#a0428e #a0428e #02bf89 #02bf89 #02bf89 "
      "#02bf89 #4d96ff #4d96ff #4d96ff #4d96ff "
      "#79490c #79490c #79490c #79490c #461ab6 "
      "#461ab6 #461ab6 #461ab6 #43d344 #43d344 "
      "#43d344 #43d344 #b06dc9 #b06dc9 #b06dc9 "
      "#b06dc9 #df4e04 #df4e04 #df4e04 #df4e04 "
      "#83d16f #83d16f #83d16f #83d16f #04f025 "
      "#04f025 #04f025 #04f025 #a2fee4 #a2fee4 "
      "#a2fee4 #a2fee4 #e80153 #e80153 #e80153 "
      "#e80153 #ffa6f8 #ffa6f8 #ffa6f8 #ffa6f8 "
      "#96f718 #96f718 #96f718 #96f718 #5e83c0 "
      "#5e83c0 #5e83c0 #5e83c0 #eb02f9 #eb02f9 "
      "#eb02f9 #eb02f9 #bf0005 #bf0005 #bf0005 "
      "#bf0005 #cfa736 #cfa736 #cfa736 #cfa736 "
      "#0a7fdf #0a7fdf #0a7fdf #0a7fdf #f685a0 "
      "#f685a0 #f685a0 #f685a0 #000a7a #000a7a "
      "#000a7a #000a7a #e12ba0 #e12ba0 #e12ba0 "
      "#e12ba0 #046372 #046372 #046372 #046372 "
      "#d1eb91 #d1eb91 #d1eb91 #d1eb91 #740377 "
      "#740377 #740377 #740377 #39d6fc #39d6fc "
      "#39d6fc #39d6fc #708e1f #708e1f #708e1f "
      "#708e1f #acb0fd #acb0fd #acb0fd #acb0fd "
      "#fd52fa #fd52fa #fd52fa #fd52fa #01840a "
      "#01840a #01840a #01840a #0626d6 #0626d6 "
      "#0626d6 #0626d6 #5c0a08 #5c0a08 #5c0a08 "
      "#5c0a08 #e3f92f #e3f92f #e3f92f #e3f92f "
      "#49ff00 #49ff00 #49ff00 #49ff00 #3ed78e "
      "#3ed78e #3ed78e #3ed78e #872cde #872cde "
      "#872cde #872cde #df5146 #df5146 #df5146 "
      "#df5146 "
};

static char ROI_32_CMD[] = {
      "ROI_i32 "
      "#43d344 #b06dc9 #df4e04 #83d16f #04f025 "
      "#a2fee4 #e80153 #ffa6f8 #96f718 #5e83c0 "
      "#eb02f9 #bf0005 #cfa736 #0a7fdf #f685a0 "
      "#000a7a #e12ba0 #046372 #d1eb91 #740377 "
      "#39d6fc #708e1f #acb0fd #fd52fa #01840a "
      "#0626d6 #5c0a08 #e3f92f #49ff00 #3ed78e "
      "#872cde #df5146 "
};

static char ROI_32_256_CMD[] = {
      "ROI_i32 "
      "#43d344 #43d344 #43d344 #43d344 #43d344 "
      "#43d344 #43d344 #43d344 #b06dc9 #b06dc9 "
      "#b06dc9 #b06dc9 #b06dc9 #b06dc9 #b06dc9 "
      "#b06dc9 #df4e04 #df4e04 #df4e04 #df4e04 "
      "#df4e04 #df4e04 #df4e04 #df4e04 #83d16f "
      "#83d16f #83d16f #83d16f #83d16f #83d16f "
      "#83d16f #83d16f #04f025 #04f025 #04f025 "
      "#04f025 #04f025 #04f025 #04f025 #04f025 "
      "#a2fee4 #a2fee4 #a2fee4 #a2fee4 #a2fee4 "
      "#a2fee4 #a2fee4 #a2fee4 #e80153 #e80153 "
      "#e80153 #e80153 #e80153 #e80153 #e80153 "
      "#e80153 #ffa6f8 #ffa6f8 #ffa6f8 #ffa6f8 "
      "#ffa6f8 #ffa6f8 #ffa6f8 #ffa6f8 #96f718 "
      "#96f718 #96f718 #96f718 #96f718 #96f718 "
      "#96f718 #96f718 #5e83c0 #5e83c0 #5e83c0 "
      "#5e83c0 #5e83c0 #5e83c0 #5e83c0 #5e83c0 "
      "#eb02f9 #eb02f9 #eb02f9 #eb02f9 #eb02f9 "
      "#eb02f9 #eb02f9 #eb02f9 #bf0005 #bf0005 "
      "#bf0005 #bf0005 #bf0005 #bf0005 #bf0005 "
      "#bf0005 #cfa736 #cfa736 #cfa736 #cfa736 "
      "#cfa736 #cfa736 #cfa736 #cfa736 #0a7fdf "
      "#0a7fdf #0a7fdf #0a7fdf #0a7fdf #0a7fdf "
      "#0a7fdf #0a7fdf #f685a0 #f685a0 #f685a0 "
      "#f685a0 #f685a0 #f685a0 #f685a0 #f685a0 "
      "#000a7a #000a7a #000a7a #000a7a #000a7a "
      "#000a7a #000a7a #000a7a #e12ba0 #e12ba0 "
      "#e12ba0 #e12ba0 #e12ba0 #e12ba0 #e12ba0 "
      "#e12ba0 #046372 #046372 #046372 #046372 "
      "#046372 #046372 #046372 #046372 #d1eb91 "
      "#d1eb91 #d1eb91 #d1eb91 #d1eb91 #d1eb91 "
      "#d1eb91 #d1eb91 #740377 #740377 #740377 "
      "#740377 #740377 #740377 #740377 #740377 "
      "#39d6fc #39d6fc #39d6fc #39d6fc #39d6fc "
      "#39d6fc #39d6fc #39d6fc #708e1f #708e1f "
      "#708e1f #708e1f #708e1f #708e1f #708e1f "
      "#708e1f #acb0fd #acb0fd #acb0fd #acb0fd "
      "#acb0fd #acb0fd #acb0fd #acb0fd #fd52fa "
      "#fd52fa #fd52fa #fd52fa #fd52fa #fd52fa "
      "#fd52fa #fd52fa #01840a #01840a #01840a "
      "#01840a #01840a #01840a #01840a #01840a "
      "#0626d6 #0626d6 #0626d6 #0626d6 #0626d6 "
      "#0626d6 #0626d6 #0626d6 #5c0a08 #5c0a08 "
      "#5c0a08 #5c0a08 #5c0a08 #5c0a08 #5c0a08 "
      "#5c0a08 #e3f92f #e3f92f #e3f92f #e3f92f "
      "#e3f92f #e3f92f #e3f92f #e3f92f #49ff00 "
      "#49ff00 #49ff00 #49ff00 #49ff00 #49ff00 "
      "#49ff00 #49ff00 #3ed78e #3ed78e #3ed78e "
      "#3ed78e #3ed78e #3ed78e #3ed78e #3ed78e "
      "#872cde #872cde #872cde #872cde #872cde "
      "#872cde #872cde #872cde #df5146 #df5146 "
      "#df5146 #df5146 #df5146 #df5146 #df5146 "
      "#df5146 "
};

/* Some maps for color blindness
Thanks to Vince Hradil who created them
based on http://bsp.pdx.edu/Publications/2006/SPM_McNames.pdf
*/
static char CB_CS_35[] = {
      "cb_spiral35  "
      "#c776ee #c776ee #c776ee #be75f1 "
      "#be75f1 #b574f5 #b574f5 #ac75f8 "
      "#ac75f8 #a276fb #a276fb #9878fd "
      "#9878fd #9878fd #8d7bfe #8d7bfe "
      "#837eff #837eff #7983ff #7983ff "
      "#6f88fe #6f88fe #658efc #658efc "
      "#658efc #5b94f9 #5b94f9 #529bf5 "
      "#529bf5 #4aa2f0 #4aa2f0 #42aaea "
      "#42aaea #3bb2e3 #3bb2e3 #3bb2e3 "
      "#34badb #34badb #2fc2d3 #2fc2d3 "
      "#2bcac9 #2bcac9 #27d2be #27d2be "
      "#25dab3 #25dab3 #24e1a7 #24e1a7 "
      "#24e1a7 #24e79a #24e79a #26ed8d "
      "#26ed8d #28f27f #28f27f #2cf672 "
      "#2cf672 #31f964 #31f964 #31f964 "
      "#37fa56 #37fa56 #3ffa49 #3ffa49 "
      "#47f83d #47f83d #50f431 #50f431 "
      "#5aef27 #5aef27 #5aef27 #64e81e "
      "#64e81e #6de016 #6de016 #77d70f "
      "#77d70f #80cd0a #80cd0a #89c306 "
      "#89c306 #89c306 #91b803 #91b803 "
      "#98ac01 #98ac01 #9fa000 #9fa000 "
      "#a49401 #a49401 #a98802 #a98802 "
      "#ad7c04 #ad7c04 #ad7c04 #b07007 "
      "#b07007 #b2640b #b2640b #b35810 "
      "#b35810 #b24d15 #b24d15 #b1431a "
      "#b1431a #b1431a #af3920 #af3920 "
      "#ac3026 #ac3026 #a8282c #a8282c "
      "#a42033 #a42033 #9e1939 #9e1939 "
      "#9e1939 #98133f #98133f #910e45 "
      "#910e45 #8a0a4a #8a0a4a #830650 "
      "#830650 #7b0354 #7b0354 #7b0354 "
};

static char CB_CS[] = {
      "cb_spiral "
      "#ffffff #fdfffc #fafff9 #f8fff5 "
      "#f6fff1 #f5feed #f4fde9 #f4fce4 "
      "#f4fbe0 #f4f9db #f4f6d7 #f5f4d3 "
      "#f6f0cf #f7edcc #f9e8c9 #fae4c6 "
      "#fbdfc4 #fdd9c2 #fed4c0 #ffcec0 "
      "#ffc7bf #ffc1bf #ffbac0 #ffb3c2 "
      "#fdacc4 #fca6c6 #f99fc9 #f799cc "
      "#f393d0 #ef8dd4 #ea88d8 #e483dc "
      "#de7fe0 #d77be5 #cf78e9 #c776ee "
      "#be75f1 #b574f5 #ac75f8 #a276fb "
      "#9878fd #8d7bfe #837eff #7983ff "
      "#6f88fe #658efc #5b94f9 #529bf5 "
      "#4aa2f0 #42aaea #3bb2e3 #34badb "
      "#2fc2d3 #2bcac9 #27d2be #25dab3 "
      "#24e1a7 #24e79a #26ed8d #28f27f "
      "#2cf672 #31f964 #37fa56 #3ffa49 "
      "#47f83d #50f431 #5aef27 #64e81e "
      "#6de016 #77d70f #80cd0a #89c306 "
      "#91b803 #98ac01 #9fa000 #a49401 "
      "#a98802 #ad7c04 #b07007 #b2640b "
      "#b35810 #b24d15 #b1431a #af3920 "
      "#ac3026 #a8282c #a42033 #9e1939 "
      "#98133f #910e45 #8a0a4a #830650 "
      "#7b0354 #720158 #6a005c #62005f "
      "#590061 #510163 #480264 #400464 "
      "#390663 #320862 #2b0a61 #240d5e "
      "#1e105b #191358 #141554 #101850 "
      "#0c1a4b #091c46 #061e41 #041f3c "
      "#022036 #012131 #00212c #002127 "
      "#002022 #001f1d #001d18 #001b14 "
      "#001810 #01160c #011209 #010f07 "
      "#010b04 #010702 #000401 #010101 "
};  /* changed last index to non-zero - not transparent */

/*
  [PT: July 4, 2022] Add these colorbars in hex here (some exist as
  floats in pbardefs_float.h); mainly for SUMA usage
*/
static char HEX_GoogleTurbo_256_CMD[] = {
    "GoogleTurbo "
    "#30123b #311542 #32184a #341b51 #351e58 #36215f #372365 #38266c "
    "#392972 #3a2c79 #3b2f7f #3c3285 #3c358b #3d3791 #3e3a96 #3f3d9c "
    "#4040a1 #4043a6 #4145ab #4148b0 #424bb5 #434eba #4350be #4353c2 "
    "#4456c7 #4458cb #455bce #455ed2 #4560d6 #4563d9 #4666dd #4668e0 "
    "#466be3 #466de6 #4670e8 #4673eb #4675ed #4678f0 #467af2 #467df4 "
    "#467ff6 #4682f8 #4584f9 #4587fb #4589fc #448cfd #438efd #4291fe "
    "#4193fe #4096fe #3f98fe #3e9bfe #3c9dfd #3ba0fc #39a2fc #38a5fb "
    "#36a8f9 #34aaf8 #33acf6 #31aff5 #2fb1f3 #2db4f1 #2bb6ef #2ab9ed "
    "#28bbeb #26bde9 #25c0e6 #23c2e4 #21c4e1 #20c6df #1ec9dc #1dcbda "
    "#1ccdd7 #1bcfd4 #1ad1d2 #19d3cf #18d5cc #18d7ca #17d9c7 #17dac4 "
    "#17dcc2 #17debf #18e0bd #18e1ba #19e3b8 #1ae4b6 #1be5b4 #1de7b1 "
    "#1ee8af #20e9ac #22eba9 #24eca6 #27eda3 #29eea0 #2cef9d #2ff09a "
    "#32f197 #35f394 #38f491 #3bf48d #3ff58a #42f687 #46f783 #4af880 "
    "#4df97c #51f979 #55fa76 #59fb72 #5dfb6f #61fc6c #65fc68 #69fd65 "
    "#6dfd62 #71fd5f #74fe5c #78fe59 #7cfe56 #80fe53 #84fe50 #87fe4d "
    "#8bfe4b #8efe48 #92fe46 #95fe44 #98fe42 #9bfd40 #9efd3e #a1fc3d "
    "#a4fc3b #a6fb3a #a9fb39 #acfa37 #aef937 #b1f836 #b3f835 #b6f735 "
    "#b9f534 #bbf434 #bef334 #c0f233 #c3f133 #c5ef33 #c8ee33 #caed33 "
    "#cdeb34 #cfea34 #d1e834 #d4e735 #d6e535 #d8e335 #dae236 #dde036 "
    "#dfde36 #e1dc37 #e3da37 #e5d838 #e7d738 #e8d538 #ead339 #ecd139 "
    "#edcf39 #efcd39 #f0cb3a #f2c83a #f3c63a #f4c43a #f6c23a #f7c039 "
    "#f8be39 #f9bc39 #f9ba38 #fab737 #fbb537 #fbb336 #fcb035 #fcae34 "
    "#fdab33 #fda932 #fda631 #fda330 #fea12f #fe9e2e #fe9b2d #fe982c "
    "#fd952b #fd9229 #fd8f28 #fd8c27 #fc8926 #fc8624 #fb8323 #fb8022 "
    "#fa7d20 #fa7a1f #f9771e #f8741c #f7711b #f76e1a #f66b18 #f56817 "
    "#f46516 #f36315 #f26014 #f15d13 #ef5a11 #ee5810 #ed550f #ec520e "
    "#ea500d #e94d0d #e84b0c #e6490b #e5460a #e3440a #e24209 #e04008 "
    "#de3e08 #dd3c07 #db3a07 #d93806 #d73606 #d63405 #d43205 #d23005 "
    "#d02f04 #ce2d04 #cb2b03 #c92903 #c72803 #c52602 #c32402 #c02302 "
    "#be2102 #bb1f01 #b91e01 #b61c01 #b41b01 #b11901 #ae1801 #ac1601 "
    "#a91501 #a61401 #a31201 #a01101 #9d1001 #9a0e01 #970d01 #940c01 "
    "#910b01 #8e0a01 #8b0901 #870801 #840701 #810602 #7d0502 #7a0402 "
};

static char HEX_Inferno_256_CMD[] = {
    "Inferno "
    "#000003 #000004 #000006 #010007 #010109 #01010b #02010e #020210 "
    "#030212 #040314 #040316 #050418 #06041b #07051d #08061f #090621 "
    "#0a0723 #0b0726 #0d0828 #0e082a #0f092d #10092f #120a32 #130a34 "
    "#140b36 #160b39 #170b3b #190b3e #1a0b40 #1c0c43 #1d0c45 #1f0c47 "
    "#200c4a #220b4c #240b4e #260b50 #270b52 #290b54 #2b0a56 #2d0a58 "
    "#2e0a5a #300a5c #32095d #34095f #350960 #370961 #390962 #3b0964 "
    "#3c0965 #3e0966 #400966 #410967 #430a68 #450a69 #460a69 #480b6a "
    "#4a0b6a #4b0c6b #4d0c6b #4f0d6c #500d6c #520e6c #530e6d #550f6d "
    "#570f6d #58106d #5a116d #5b116e #5d126e #5f126e #60136e #62146e "
    "#63146e #65156e #66156e #68166e #6a176e #6b176e #6d186e #6e186e "
    "#70196e #72196d #731a6d #751b6d #761b6d #781c6d #7a1c6d #7b1d6c "
    "#7d1d6c #7e1e6c #801f6b #811f6b #83206b #85206a #86216a #88216a "
    "#892269 #8b2269 #8d2369 #8e2468 #902468 #912567 #932567 #952666 "
    "#962666 #982765 #992864 #9b2864 #9c2963 #9e2963 #a02a62 #a12b61 "
    "#a32b61 #a42c60 #a62c5f #a72d5f #a92e5e #ab2e5d #ac2f5c #ae305b "
    "#af315b #b1315a #b23259 #b43358 #b53357 #b73456 #b83556 #ba3655 "
    "#bb3754 #bd3753 #be3852 #bf3951 #c13a50 #c23b4f #c43c4e #c53d4d "
    "#c73e4c #c83e4b #c93f4a #cb4049 #cc4148 #cd4247 #cf4446 #d04544 "
    "#d14643 #d24742 #d44841 #d54940 #d64a3f #d74b3e #d94d3d #da4e3b "
    "#db4f3a #dc5039 #dd5238 #de5337 #df5436 #e05634 #e25733 #e35832 "
    "#e45a31 #e55b30 #e65c2e #e65e2d #e75f2c #e8612b #e9622a #ea6428 "
    "#eb6527 #ec6726 #ed6825 #ed6a23 #ee6c22 #ef6d21 #f06f1f #f0701e "
    "#f1721d #f2741c #f2751a #f37719 #f37918 #f47a16 #f57c15 #f57e14 "
    "#f68012 #f68111 #f78310 #f7850e #f8870d #f8880c #f88a0b #f98c09 "
    "#f98e08 #f99008 #fa9107 #fa9306 #fa9506 #fa9706 #fb9906 #fb9b06 "
    "#fb9d06 #fb9e07 #fba007 #fba208 #fba40a #fba60b #fba80d #fbaa0e "
    "#fbac10 #fbae12 #fbb014 #fbb116 #fbb318 #fbb51a #fbb71c #fbb91e "
    "#fabb21 #fabd23 #fabf25 #fac128 #f9c32a #f9c52c #f9c72f #f8c931 "
    "#f8cb34 #f8cd37 #f7cf3a #f7d13c #f6d33f #f6d542 #f5d745 #f5d948 "
    "#f4db4b #f4dc4f #f3de52 #f3e056 #f3e259 #f2e45d #f2e660 #f1e864 "
    "#f1e968 #f1eb6c #f1ed70 #f1ee74 #f1f079 #f1f27d #f2f381 #f2f485 "
    "#f3f689 #f4f78d #f5f891 #f6fa95 #f7fb99 #f9fc9d #fafda0 #fcfea4 "
};

static char HEX_Magma_256_CMD[] = {
    "Magma "
    "#000003 #000004 #000006 #010007 #010109 #01010b #02020d #02020f "
    "#030311 #040313 #040415 #050417 #060519 #07051b #08061d #09071f "
    "#0a0722 #0b0824 #0c0926 #0d0a28 #0e0a2a #0f0b2c #100c2f #110c31 "
    "#120d33 #140d35 #150e38 #160e3a #170f3c #180f3f #1a1041 #1b1044 "
    "#1c1046 #1e1049 #1f114b #20114d #221150 #231152 #251155 #261157 "
    "#281159 #2a115c #2b115e #2d1060 #2f1062 #301065 #321067 #341068 "
    "#350f6a #370f6c #390f6e #3b0f6f #3c0f71 #3e0f72 #400f73 #420f74 "
    "#430f75 #450f76 #470f77 #481078 #4a1079 #4b1079 #4d117a #4f117b "
    "#50127b #52127c #53137c #55137d #57147d #58157e #5a157e #5b167e "
    "#5d177e #5e177f #60187f #61187f #63197f #651a80 #661a80 #681b80 "
    "#691c80 #6b1c80 #6c1d80 #6e1e81 #6f1e81 #711f81 #731f81 #742081 "
    "#762181 #772181 #792281 #7a2281 #7c2381 #7e2481 #7f2481 #812581 "
    "#822581 #842681 #852681 #872781 #892881 #8a2881 #8c2980 #8d2980 "
    "#8f2a80 #912a80 #922b80 #942b80 #952c80 #972c7f #992d7f #9a2d7f "
    "#9c2e7f #9e2e7e #9f2f7e #a12f7e #a3307e #a4307d #a6317d #a7317d "
    "#a9327c #ab337c #ac337b #ae347b #b0347b #b1357a #b3357a #b53679 "
    "#b63679 #b83778 #b93778 #bb3877 #bd3977 #be3976 #c03a75 #c23a75 "
    "#c33b74 #c53c74 #c63c73 #c83d72 #ca3e72 #cb3e71 #cd3f70 #ce4070 "
    "#d0416f #d1426e #d3426d #d4436d #d6446c #d7456b #d9466a #da4769 "
    "#dc4869 #dd4968 #de4a67 #e04b66 #e14c66 #e24d65 #e44e64 #e55063 "
    "#e65162 #e75262 #e85461 #ea5560 #eb5660 #ec585f #ed595f #ee5b5e "
    "#ee5d5d #ef5e5d #f0605d #f1615c #f2635c #f3655c #f3675b #f4685b "
    "#f56a5b #f56c5b #f66e5b #f6705b #f7715b #f7735c #f8755c #f8775c "
    "#f9795c #f97b5d #f97d5d #fa7f5e #fa805e #fa825f #fb8460 #fb8660 "
    "#fb8861 #fb8a62 #fc8c63 #fc8e63 #fc9064 #fc9265 #fc9366 #fd9567 "
    "#fd9768 #fd9969 #fd9b6a #fd9d6b #fd9f6c #fda16e #fda26f #fda470 "
    "#fea671 #fea873 #feaa74 #feac75 #feae76 #feaf78 #feb179 #feb37b "
    "#feb57c #feb77d #feb97f #febb80 #febc82 #febe83 #fec085 #fec286 "
    "#fec488 #fec689 #fec78b #fec98d #fecb8e #fdcd90 #fdcf92 #fdd193 "
    "#fdd295 #fdd497 #fdd698 #fdd89a #fdda9c #fddc9d #fddd9f #fddfa1 "
    "#fde1a3 #fce3a5 #fce5a6 #fce6a8 #fce8aa #fceaac #fcecae #fceeb0 "
    "#fcf0b1 #fcf1b3 #fcf3b5 #fcf5b7 #fbf7b9 #fbf9bb #fbfabd #fbfcbf "
};

static char HEX_Plasma_256_CMD[] = {
    "Plasma "
    "#0c0786 #100787 #130689 #15068a #18068b #1b068c #1d068d #1f058e "
    "#21058f #230590 #250591 #270592 #290593 #2b0594 #2d0494 #2f0495 "
    "#310496 #330497 #340498 #360498 #380499 #3a049a #3b039a #3d039b "
    "#3f039c #40039c #42039d #44039e #45039e #47029f #49029f #4a02a0 "
    "#4c02a1 #4e02a1 #4f02a2 #5101a2 #5201a3 #5401a3 #5601a3 #5701a4 "
    "#5901a4 #5a00a5 #5c00a5 #5e00a5 #5f00a6 #6100a6 #6200a6 #6400a7 "
    "#6500a7 #6700a7 #6800a7 #6a00a7 #6c00a8 #6d00a8 #6f00a8 #7000a8 "
    "#7200a8 #7300a8 #7500a8 #7601a8 #7801a8 #7901a8 #7b02a8 #7c02a7 "
    "#7e03a7 #7f03a7 #8104a7 #8204a7 #8405a6 #8506a6 #8607a6 #8807a5 "
    "#8908a5 #8b09a4 #8c0aa4 #8e0ca4 #8f0da3 #900ea3 #920fa2 #9310a1 "
    "#9511a1 #9612a0 #9713a0 #99149f #9a159e #9b179e #9d189d #9e199c "
    "#9f1a9b #a01b9b #a21c9a #a31d99 #a41e98 #a51f97 #a72197 #a82296 "
    "#a92395 #aa2494 #ac2593 #ad2692 #ae2791 #af2890 #b02a8f #b12b8f "
    "#b22c8e #b42d8d #b52e8c #b62f8b #b7308a #b83289 #b93388 #ba3487 "
    "#bb3586 #bc3685 #bd3784 #be3883 #bf3982 #c03b81 #c13c80 #c23d80 "
    "#c33e7f #c43f7e #c5407d #c6417c #c7427b #c8447a #c94579 #ca4678 "
    "#cb4777 #cc4876 #cd4975 #ce4a75 #cf4b74 #d04d73 #d14e72 #d14f71 "
    "#d25070 #d3516f #d4526e #d5536d #d6556d #d7566c #d7576b #d8586a "
    "#d95969 #da5a68 #db5b67 #dc5d66 #dc5e66 #dd5f65 #de6064 #df6163 "
    "#df6262 #e06461 #e16560 #e26660 #e3675f #e3685e #e46a5d #e56b5c "
    "#e56c5b #e66d5a #e76e5a #e87059 #e87158 #e97257 #ea7356 #ea7455 "
    "#eb7654 #ec7754 #ec7853 #ed7952 #ed7b51 #ee7c50 #ef7d4f #ef7e4e "
    "#f0804d #f0814d #f1824c #f2844b #f2854a #f38649 #f38748 #f48947 "
    "#f48a47 #f58b46 #f58d45 #f68e44 #f68f43 #f69142 #f79241 #f79341 "
    "#f89540 #f8963f #f8983e #f9993d #f99a3c #fa9c3b #fa9d3a #fa9f3a "
    "#faa039 #fba238 #fba337 #fba436 #fca635 #fca735 #fca934 #fcaa33 "
    "#fcac32 #fcad31 #fdaf31 #fdb030 #fdb22f #fdb32e #fdb52d #fdb62d "
    "#fdb82c #fdb92b #fdbb2b #fdbc2a #fdbe29 #fdc029 #fdc128 #fdc328 "
    "#fdc427 #fdc626 #fcc726 #fcc926 #fccb25 #fccc25 #fcce25 #fbd024 "
    "#fbd124 #fbd324 #fad524 #fad624 #fad824 #f9d924 #f9db24 #f8dd24 "
    "#f8df24 #f7e024 #f7e225 #f6e425 #f6e525 #f5e726 #f5e926 #f4ea26 "
    "#f3ec26 #f3ee26 #f2f026 #f2f126 #f1f326 #f0f525 #f0f623 #eff821 "
};

static char HEX_Viridis_256_CMD[] = {
    "Viridis "
    "#440154 #440255 #440357 #450558 #45065a #45085b #46095c #460b5e "
    "#460c5f #460e61 #470f62 #471163 #471265 #471466 #471567 #471669 "
    "#47186a #48196b #481a6c #481c6e #481d6f #481e70 #482071 #482172 "
    "#482273 #482374 #472575 #472676 #472777 #472878 #472a79 #472b7a "
    "#472c7b #462d7c #462f7c #46307d #46317e #45327f #45347f #453580 "
    "#453681 #443781 #443982 #433a83 #433b83 #433c84 #423d84 #423e85 "
    "#424085 #414186 #414286 #404387 #404487 #3f4587 #3f4788 #3e4888 "
    "#3e4989 #3d4a89 #3d4b89 #3d4c89 #3c4d8a #3c4e8a #3b508a #3b518a "
    "#3a528b #3a538b #39548b #39558b #38568b #38578c #37588c #37598c "
    "#365a8c #365b8c #355c8c #355d8c #345e8d #345f8d #33608d #33618d "
    "#32628d #32638d #31648d #31658d #31668d #30678d #30688d #2f698d "
    "#2f6a8d #2e6b8e #2e6c8e #2e6d8e #2d6e8e #2d6f8e #2c708e #2c718e "
    "#2c728e #2b738e #2b748e #2a758e #2a768e #2a778e #29788e #29798e "
    "#287a8e #287a8e #287b8e #277c8e #277d8e #277e8e #267f8e #26808e "
    "#26818e #25828e #25838d #24848d #24858d #24868d #23878d #23888d "
    "#23898d #22898d #228a8d #228b8d #218c8d #218d8c #218e8c #208f8c "
    "#20908c #20918c #1f928c #1f938b #1f948b #1f958b #1f968b #1e978a "
    "#1e988a #1e998a #1e998a #1e9a89 #1e9b89 #1e9c89 #1e9d88 #1e9e88 "
    "#1e9f88 #1ea087 #1fa187 #1fa286 #1fa386 #20a485 #20a585 #21a685 "
    "#21a784 #22a784 #23a883 #23a982 #24aa82 #25ab81 #26ac81 #27ad80 "
    "#28ae7f #29af7f #2ab07e #2bb17d #2cb17d #2eb27c #2fb37b #30b47a "
    "#32b57a #33b679 #35b778 #36b877 #38b976 #39b976 #3bba75 #3dbb74 "
    "#3ebc73 #40bd72 #42be71 #44be70 #45bf6f #47c06e #49c16d #4bc26c "
    "#4dc26b #4fc369 #51c468 #53c567 #55c666 #57c665 #59c764 #5bc862 "
    "#5ec961 #60c960 #62ca5f #64cb5d #67cc5c #69cc5b #6bcd59 #6dce58 "
    "#70ce56 #72cf55 #74d054 #77d052 #79d151 #7cd24f #7ed24e #81d34c "
    "#83d34b #86d449 #88d547 #8bd546 #8dd644 #90d643 #92d741 #95d73f "
    "#97d83e #9ad83c #9dd93a #9fd938 #a2da37 #a5da35 #a7db33 #aadb32 "
    "#addc30 #afdc2e #b2dd2c #b5dd2b #b7dd29 #bade27 #bdde26 #bfdf24 "
    "#c2df22 #c5df21 #c7e01f #cae01e #cde01d #cfe11c #d2e11b #d4e11a "
    "#d7e219 #dae218 #dce218 #dfe318 #e1e318 #e4e318 #e7e419 #e9e419 "
    "#ece41a #eee51b #f1e51c #f3e51e #f6e61f #f8e621 #fae622 #fde724 "
};


static char HEX_CET_L17_256_CMD[] = {
    "CET_L17 "
    "#feffff #fefefc #fefdf9 #fdfcf6 #fdfbf3 #fcfbf0 #fcfaee #fbf9eb "
    "#fbf8e8 #faf8e5 #faf7e2 #f9f6e0 #f9f5dd #f8f4da #f7f4d7 #f7f3d4 "
    "#f6f2d2 #f6f1cf #f5f1cc #f4f0c9 #f4efc7 #f3eec4 #f3eec1 #f2edbe "
    "#f2ecbc #f2ebbb #f2eab9 #f2e9b7 #f2e8b6 #f2e7b4 #f2e6b2 #f2e5b1 "
    "#f2e4af #f2e2ad #f2e1ac #f2e0aa #f2dfa8 #f2dea7 #f2dda5 #f2dca3 "
    "#f2dba2 #f2daa0 #f2d99e #f2d89d #f2d79b #f2d699 #f1d598 #f2d496 "
    "#f2d395 #f2d194 #f2d093 #f2cf92 #f2ce90 #f2cd8f #f2cc8e #f3cb8d "
    "#f3c98c #f3c88a #f3c789 #f3c688 #f3c587 #f3c486 #f3c284 #f3c183 "
    "#f3c082 #f3bf81 #f3be80 #f4bd7e #f4bb7d #f4ba7c #f4b97b #f4b87a "
    "#f4b77a #f4b579 #f4b478 #f4b378 #f4b277 #f4b076 #f4af75 #f4ae75 "
    "#f4ad74 #f4ac73 #f5aa73 #f5a972 #f5a871 #f5a771 #f5a570 #f5a46f "
    "#f5a36f #f5a16e #f5a06d #f59f6c #f59e6c #f59c6b #f59b6b #f59a6b "
    "#f5996b #f5976a #f4966a #f4956a #f4936a #f4926a #f49169 #f49069 "
    "#f48e69 #f48d69 #f48c68 #f48a68 #f48968 #f38868 #f38668 #f38567 "
    "#f38467 #f38267 #f38167 #f38066 #f37e66 #f27d66 #f27c67 #f27a67 "
    "#f17967 #f17867 #f17767 #f07568 #f07468 #f07368 #ef7168 #ef7068 "
    "#ef6f69 #ee6d69 #ee6c69 #ee6b69 #ed6969 #ed6869 #ed6669 #ec656a "
    "#ec646a #ec626a #eb616a #eb5f6a #ea5e6b #ea5d6b #e95c6c #e95a6c "
    "#e8596d #e7586d #e7576d #e6556e #e5546e #e5536f #e4516f #e45070 "
    "#e34f70 #e24d71 #e24c71 #e14b71 #e04972 #e04872 #df4773 #de4573 "
    "#de4474 #dd4274 #dc4174 #db4075 #da3f76 #d93e76 #d83c77 #d73b77 "
    "#d63a78 #d53979 #d43879 #d3377a #d2367b #d1357b #d0337c #cf327c "
    "#ce317d #cd307d #cc2f7e #cb2d7f #ca2c7f #c92b80 #c82a80 #c72881 "
    "#c62782 #c52682 #c42583 #c22483 #c12484 #bf2385 #be2285 #bd2286 "
    "#bb2187 #ba2187 #b82088 #b71f89 #b51f89 #b41e8a #b21e8b #b11d8b "
    "#af1c8c #ae1c8d #ac1b8d #ab1b8e #a91a8f #a8198f #a61990 #a41890 "
    "#a31891 #a11892 #9f1892 #9d1893 #9b1993 #991994 #971995 #951a95 "
    "#931a96 #911a96 #8f1b97 #8d1b97 #8b1b98 #891c99 #871c99 #851c9a "
    "#821d9a #801d9b #7e1d9b #7c1d9c #791e9d #771e9d #751e9e #721e9e "
    "#701f9f #6d209f #6a20a0 #6721a0 #6522a0 #6222a1 #5e23a1 #5b23a1 "
    "#5824a2 #5524a2 #5125a2 #4e25a3 #4a26a3 #4626a4 #4227a4 #3e27a4 "
    "#3927a5 #3428a5 #2e28a5 #2828a6 #2129a6 #1829a6 #0929a7 #002aa7 "
};

// [PT: Aug 16, 2017] New(ish) colorbar.  Took "Reds_and_Blues",
// inverted each of the halves individually, and made this.
static char R_AND_B_INV_256_CMD[] = {
"Reds_and_Blues_Inv "
"#ff0b00 "
"#ff0e00 "
"#ff1100 "
"#ff1400 "
"#ff1700 "
"#ff1a00 "
"#ff1d00 "
"#ff2000 "
"#ff2200 "
"#ff2500 "
"#ff2700 "
"#ff2a00 "
"#ff2c00 "
"#ff2f00 "
"#ff3100 "
"#ff3400 "
"#ff3600 "
"#ff3800 "
"#ff3a00 "
"#ff3d00 "
"#ff3f00 "
"#ff4100 "
"#ff4300 "
"#ff4600 "
"#ff4800 "
"#ff4a00 "
"#ff4c00 "
"#ff4e00 "
"#ff5000 "
"#ff5200 "
"#ff5400 "
"#ff5700 "
"#ff5900 "
"#ff5b00 "
"#ff5d00 "
"#ff5f00 "
"#ff6100 "
"#ff6300 "
"#ff6500 "
"#ff6700 "
"#ff6900 "
"#ff6b00 "
"#ff6d00 "
"#ff6e00 "
"#ff7000 "
"#ff7200 "
"#ff7400 "
"#ff7600 "
"#ff7800 "
"#ff7a00 "
"#ff7c00 "
"#ff7e00 "
"#ff8000 "
"#ff8200 "
"#ff8300 "
"#ff8500 "
"#ff8700 "
"#ff8900 "
"#ff8b00 "
"#ff8d00 "
"#ff8e00 "
"#ff9000 "
"#ff9200 "
"#ff9400 "
"#ff9600 "
"#ff9700 "
"#ff9900 "
"#ff9b00 "
"#ff9d00 "
"#ff9f00 "
"#ffa000 "
"#ffa200 "
"#ffa400 "
"#ffa600 "
"#ffa700 "
"#ffa900 "
"#ffab00 "
"#ffad00 "
"#ffae00 "
"#ffb000 "
"#ffb200 "
"#ffb400 "
"#ffb500 "
"#ffb700 "
"#ffb900 "
"#ffba00 "
"#ffbc00 "
"#ffbe00 "
"#ffbf00 "
"#ffc100 "
"#ffc300 "
"#ffc500 "
"#ffc600 "
"#ffc800 "
"#ffca00 "
"#ffcb00 "
"#ffcd00 "
"#ffcf00 "
"#ffd000 "
"#ffd200 "
"#ffd400 "
"#ffd500 "
"#ffd700 "
"#ffd800 "
"#ffda00 "
"#ffdc00 "
"#ffdd00 "
"#ffdf00 "
"#ffe100 "
"#ffe200 "
"#ffe400 "
"#ffe500 "
"#ffe700 "
"#ffe900 "
"#ffea00 "
"#ffec00 "
"#ffee00 "
"#ffef00 "
"#fff100 "
"#fff200 "
"#fff400 "
"#fff600 "
"#fff700 "
"#fff900 "
"#fffa00 "
"#fffc00 "
"#fffd00 "
"#ffff00 "
"#00f2ff "
"#00f1ff "
"#00efff "
"#00edff "
"#00ecff "
"#00eaff "
"#00e9ff "
"#00e7ff "
"#00e5ff "
"#00e4ff "
"#00e2ff "
"#00e0ff "
"#00dfff "
"#00ddff "
"#00dbff "
"#00daff "
"#00d8ff "
"#00d6ff "
"#00d5ff "
"#00d3ff "
"#00d1ff "
"#00d0ff "
"#00ceff "
"#00ccff "
"#00cbff "
"#00c9ff "
"#00c7ff "
"#00c6ff "
"#00c4ff "
"#00c2ff "
"#00c1ff "
"#00bfff "
"#00bdff "
"#00bcff "
"#00baff "
"#00b8ff "
"#00b6ff "
"#00b5ff "
"#00b3ff "
"#00b1ff "
"#00afff "
"#00aeff "
"#00acff "
"#00aaff "
"#00a8ff "
"#00a7ff "
"#00a5ff "
"#00a3ff "
"#00a1ff "
"#00a0ff "
"#009eff "
"#009cff "
"#009aff "
"#0098ff "
"#0097ff "
"#0095ff "
"#0093ff "
"#0091ff "
"#008fff "
"#008dff "
"#008cff "
"#008aff "
"#0088ff "
"#0086ff "
"#0084ff "
"#0082ff "
"#0080ff "
"#007fff "
"#007dff "
"#007bff "
"#0079ff "
"#0077ff "
"#0075ff "
"#0073ff "
"#0071ff "
"#006fff "
"#006dff "
"#006bff "
"#0069ff "
"#0067ff "
"#0065ff "
"#0063ff "
"#0061ff "
"#005fff "
"#005dff "
"#005bff "
"#0059ff "
"#0057ff "
"#0055ff "
"#0053ff "
"#0051ff "
"#004fff "
"#004cff "
"#004aff "
"#0048ff "
"#0046ff "
"#0044ff "
"#0042ff "
"#003fff "
"#003dff "
"#003bff "
"#0038ff "
"#0036ff "
"#0034ff "
"#0031ff "
"#002fff "
"#002dff "
"#002aff "
"#0028ff "
"#0025ff "
"#0022ff "
"#0020ff "
"#001dff "
"#001aff "
"#0017ff "
"#0015ff "
"#0012ff "
"#000eff "
"#0b00ff "
"#0e00ff "
"#1200ff "
"#1500ff "
"#1700ff "
"#1a00ff "
"#1d00ff "
"#2000ff "
"#2200ff "
"#2500ff "
};

/* regular gray from white to black */
static char GRAY_CS[] = {
   "gray_scale "
   "#ffffff #010101 "
};

/* mirrored gray for +/- values */
static char GRAY_CIRCLE_CS[] = {
   "gray_circle "
   "#ffffff #010101 #ffffff "
};
/* mirrored and inverted gray for +/- values */
static char GRAY_INV_CIRCLE_CS[] = {
   "inverted_gray_circle "
   "#010101 #ffffff #010101 "
};

/* amber monochrome from bright orange to black */
static char AMBER_CS[] = {
   "amber_monochrome "
   "#ffbf00 #010100 "
};

/* mirrored amber for +/- values */
static char AMBER_CIRCLE_CS[] = {
   "amber_circle "
   "#ffbf00 #010100 #ffbf00 "
};

/* mirrored and inverted amber for +/- values */
static char AMBER_INV_CIRCLE_CS[] = {
   "inverted_amber_circle "
   "#010100 #ffbf00 #010100 "
};

/* mirrored and inverted amber for +/- values */
static char AMBER_REDTOP_BLUEBOT_CS[] = {
   "amber_redtop_bluebot "
" #ff0000 "
" #febe00 #fdbd00 #fcbd00 #fbbc00 "
" #fabb00 #f9ba00 #f8ba00 #f7b900 "
" #f6b800 #f5b700 #f4b700 #f3b600 "
" #f2b500 #f1b400 #f0b400 #efb300 "
" #eeb200 #edb100 #ecb100 #ebb000 "
" #eaaf00 #e9ae00 #e8ae00 #e7ad00 "
" #e6ac00 #e5ab00 #e4ab00 #e3aa00 "
" #e2a900 #e1a900 #e0a800 #dfa700 "
" #dea600 #dda600 #dca500 #dba400 "
" #daa300 #d9a300 #d8a200 #d7a100 "
" #d6a000 #d5a000 #d49f00 #d39e00 "
" #d29d00 #d19d00 #d09c00 #cf9b00 "
" #ce9a00 #cd9a00 #cc9900 #cb9800 "
" #ca9700 #c99700 #c89600 #c79500 "
" #c69400 #c59400 #c49300 #c39200 "
" #c29200 #c19100 #c09000 #bf8f00 "
" #bf8f00 #be8e00 #bd8d00 #bc8c00 "
" #bb8c00 #ba8b00 #b98a00 #b88900 "
" #b78900 #b68800 #b58700 #b48600 "
" #b38600 #b28500 #b18400 #b08300 "
" #af8300 #ae8200 #ad8100 #ac8000 "
" #ab8000 #aa7f00 #a97e00 #a87e00 "
" #a77d00 #a67c00 #a57b00 #a47b00 "
" #a37a00 #a27900 #a17800 #a07800 "
" #9f7700 #9e7600 #9d7500 #9c7500 "
" #9b7400 #9a7300 #997200 #987200 "
" #977100 #967000 #956f00 #946f00 "
" #936e00 #926d00 #916c00 #906c00 "
" #8f6b00 #8e6a00 #8d6a00 #8c6900 "
" #8b6800 #8a6700 #896700 #886600 "
" #876500 #866400 #856400 #846300 "
" #836200 #826100 #816100 #806000 "
" #7f5f00 #7e5e00 #7d5e00 #7c5d00 "
" #7b5c00 #7a5b00 #795b00 #785a00 "
" #775900 #765800 #755800 #745700 "
" #735600 #725500 #715500 #705400 "
" #6f5300 #6e5300 #6d5200 #6c5100 #6b5000 "
" #6a5000 #694f00 #684e00 #674d00 #664d00 "
" #654c00 #644b00 #634a00 #624a00 #614900 "
" #604800 #5f4700 #5e4700 #5d4600 #5c4500 "
" #5b4400 #5a4400 #594300 #584200 #574100 "
" #564100 #554000 #543f00 #533f00 #523e00 "
" #513d00 #503c00 #4f3c00 #4e3b00 #4d3a00 "
" #4c3900 #4b3900 #4a3800 #493700 #483600 "
" #473600 #463500 #453400 #443300 #433300 "
" #423200 #413100 #403000 #403000 #3f2f00 "
" #3e2e00 #3d2d00 #3c2d00 #3b2c00 #3a2b00 "
" #392b00 #382a00 #372900 #362800 #352800 "
" #342700 #332600 #322500 #312500 #302400 "
" #2f2300 #2e2200 #2d2200 #2c2100 #2b2000 "
" #2a1f00 #291f00 #281e00 #271d00 #261c00 "
" #251c00 #241b00 #231a00 #221900 #211900 "
" #201800 #1f1700 #1e1600 #1d1600 #1c1500 "
" #1b1400 #1a1400 #191300 #181200 #171100 "
" #161100 #151000 #140f00 #130e00 #120e00 "
" #110d00 #100c00 #0f0b00 #0e0b00 #0d0a00 "
" #0c0900 #0b0800 #0a0800 #090700 #080600 "
" #070500 #060500 #050400 #040300 "
" #0000ff "
};

/* green monochrome from bright green to black */
static char GREEN_CS[] = {
   "green_monochrome "
   "#00ff00 #000100 "
};

/* blue monochrome from bright blue to black  - really "azure" */
static char BLUE_CS[] = {
   "blue_monochrome "
   "#007fff #000001 "
};

/* red monochrome from bright red to black */
static char RED_CS[] = {
   "red_monochrome "
   "#ff0000 #010000 "
};

/* amber variant for @AddEdge */
static char ADD_EDGE[] = {                /* generated with: MakeColorMap -fn  */
   "Add_Edge "                            /*    Fiducials_Ncol -ahc Add_Edge   */
   "#00FFFF #66FF00 #00FF00 #660000 "     /*    -flipud  */
   "#FF0000 #FFBF00 #FEBE00 #FDBD00 "     /* with Fiducials_Ncol containing */
   "#FBBD00 #FABC00 #F9BB00 #F8BA00 "     /* 0.4860    0.8913    0.7621   0 */
   "#F7B900 #F5B800 #F4B700 #F3B600 "     /* 0.7   0     0.7   1 */
   "#F2B500 #F0B400 #EFB400 #EEB300 "     /* 0.4514    0.0439    0.0272 2 */
   "#EDB200 #ECB100 #EAB000 #E9AF00 "     /* 0.0093    0.9400    0.5966   3 */
   "#E8AE00 #E7AD00 #E6AC00 #E4AB00 "     /* 0.9810    0.0697    0.0450 4  */
   "#E3AB00 #E2AA00 #E1A900 #DFA800 "     /* 0.13     0.10  0.0   5  */
   "#DEA700 #DDA600 #DCA500 #DBA400 "     /* 0.42     0.32  0     128 */
   "#D9A300 #D8A200 #D7A200 #D6A100 "     /* 1     0.75  0     250 */
   "#D5A000 #D39F00 #D29E00 #D19D00 "     /* 1     0     0     251 */
   "#D09C00 #CF9B00 #CD9A00 #CC9A00 "     /* 0.0   0.0   0.4     252 */
   "#CB9900 #CA9800 #C89700 #C79600 "     /* 0     1     0     253 */
   "#C69500 #C59400 #C49300 #C29200 "     /* 0.4   1     0     254 */
   "#C19100 #C09100 #BF9000 #BE8F00 "     /* 0     1     1     255  */
   "#BC8E00 #BB8D00 #BA8C00 #B98B00 "
   "#B78A00 #B68900 #B58800 #B48800 "
   "#B38700 #B18600 #B08500 #AF8400 "
   "#AE8300 #AD8200 #AB8100 #AA8000 "
   "#A97F00 #A87F00 #A77E00 #A57D00 "
   "#A47C00 #A37B00 #A27A00 #A07900 "
   "#9F7800 #9E7700 #9D7600 #9C7600 "
   "#9A7500 #997400 #987300 #977200 "
   "#967100 #947000 #936F00 #926E00 "
   "#916D00 #8F6D00 #8E6C00 #8D6B00 "
   "#8C6A00 #8B6900 #896800 #886700 "
   "#876600 #866500 #856400 #836400 "
   "#826300 #816200 #806100 #7E6000 "
   "#7D5F00 #7C5E00 #7B5D00 #7A5C00 "
   "#785B00 #775B00 #765A00 #755900 "
   "#745800 #725700 #715600 #705500 "
   "#6F5400 #6E5300 #6C5200 #6B5200 "
   "#6A5100 #6A5100 #695000 #695000 "
   "#684F00 #674F00 #674E00 #664E00 "
   "#664D00 #654D00 #644D00 #644C00 "
   "#634C00 #634B00 #624B00 #614A00 "
   "#614A00 #604900 #604900 #5F4800 "
   "#5E4800 #5E4800 #5D4700 #5D4700 "
   "#5C4600 #5B4600 #5B4500 #5A4500 "
   "#5A4400 #594400 #584300 #584300 "
   "#574300 #574200 #564200 #554100 "
   "#554100 #544000 #544000 #533F00 "
   "#523F00 #523E00 #513E00 #513E00 "
   "#503D00 #4F3D00 #4F3C00 #4E3C00 "
   "#4E3B00 #4D3B00 #4C3A00 #4C3A00 "
   "#4B3900 #4B3900 #4A3900 #493800 "
   "#493800 #483700 #483700 #473600 "
   "#463600 #463500 #453500 #453400 "
   "#443400 #433300 #433300 #423300 "
   "#423200 #413200 #403100 #403100 "
   "#3F3000 #3F3000 #3E2F00 #3D2F00 "
   "#3D2E00 #3C2E00 #3C2E00 #3B2D00 "
   "#3A2D00 #3A2C00 #392C00 #392B00 "
   "#382B00 #372A00 #372A00 #362900 "
   "#362900 #352900 #342800 #342800 "
   "#332700 #332700 #322600 #312600 "
   "#312500 #302500 #302400 #2F2400 "
   "#2E2400 #2E2300 #2D2300 #2D2200 "
   "#2C2200 #2B2100 #2B2100 #2A2000 "
   "#2A2000 #291F00 #281F00 #281F00 "
   "#271E00 #271E00 #261D00 #251D00 "
   "#251C00 #241C00 #241B00 #231B00 "
   "#221A00 #221A00 #211A00 #FA120B "
   "#02F098 #730B07 #B200B2 #7CE3C2 "
};

/* FreeSurfer colormap from 0 to 255
Created from FreeSurferColorLUT.txt
this way:
MakeColorMap   -fscolut 0 255 \
               -ahc FreeSurfer_255 \
               -flipud > fs_pal.pal
Open fs_pal.pal, remove the zeroth color
   (all 0 #000000 ) from the bottom
Add _CMD (indicating direct color
mapping needed )to var name and add
  entry in pbar.c
*/
static char FREESURFER_SEG_255_CMD[]={
   "FreeSurfer_Seg_i255 "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #C596FA #C53AFA #CD3F4E "
   "#00E200 #68FFFF #808080 #FFCC66 "
   "#8020FF #800000 #791188 #CC99CC "
   "#80FF80 #20C8FF #C4A080 #008000 "
   "#8080FF #FF0000 #0000FF #400040 "
   "#FFFF00 #FF8080 #FFCC99 #003280 "
   "#806040 #007E4B #80C4A4 #0064FF "
   "#C4C400 #FFA4A4 #00C4FF #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #EE3B3B #8B795E #EEEED1 "
   "#CDCD00 #FF0000 #E0E0E0 #8B795E "
   "#4682B5 #4682B4 #EFA0DF #34D1E2 "
   "#FFFFFE #DDE244 #8B8B8B #8604A0 "
   "#418714 #418714 #152784 #152784 "
   "#483D8B #FF0000 #EEAEEE #DD27C8 "
   "#EE3B3B #9313AD #1E90FF #228B22 "
   "#1AED39 #85CBE5 #ADFF2F #00008B "
   "#EE0000 #4AFF4A #C8C8C8 #EEEED1 "
   "#CDB38B #FF1493 #7C8CB3 #7C90B2 "
   "#7C8FB2 #7C8EB2 #7C8DB2 #7F8CB2 "
   "#7E8CB2 #7D8CB2 #7C8CB2 #7C8CB3 "
   "#7C90B2 #7C8FB2 #7C8EB2 #7C8DB2 "
   "#7F8CB2 #7E8CB2 #7D8CB2 #7C8CB2 "
   "#8B8B8B #A020F0 #CD0A7D #CD0A7D "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #8B8B8B #8B8B8B #8B8B8B "
   "#8B8B8B #FAFF32 #EAA91E #FFDAB9 "
   "#FFDAB9 #A46CE2 #A46CE2 #A46CE2 "
   "#FF940A #FF940A #C846FF #781286 "
   "#781286 #7A8732 #7A8732 #78BE96 "
   "#7A8732 #4A9B3C #783E2B #4A9B3C "
   "#333287 #7A8732 #87324A #643264 "
   "#00C8DD #A020F0 #87CEEB #A52A2A "
   "#00FF7F #FFA500 #FFA500 #3C3AD2 "
   "#50C462 #67FFFF #DCD814 #0D30FF "
   "#EC0DB0 #7ABADC #00760E #00760E "
   "#E69422 #DCF8A4 #009400 #C43AFA "
   "#781286 #CD3E4E #00E100 #CD3E4E "
   "#7A8732 #4A9B3C #783E2B #4A9B3C "
   "#333287 #7A8732 #87324A #643264 "
   "#00C8C8 #A020F0 #87CEEB #A52A2A "
   "#00FF7F #FFA500 #FFA500 #3C3C3C "
   "#3C3AD2 #3C3AD2 #3C3AD2 #3C3AD2 "
   "#50C462 #67FFFF #DCD814 #779FB0 "
   "#2ACCA4 #CCB68E #0C30FF #EC0DB0 "
   "#7ABADC #00760E #00760E #E69422 "
   "#DCF8A4 #009400 #C43AFA #781286 "
   "#CD3E4E #F5F5F5 #CD3E4E "
};

static char RedBlueGreen_CMD[] = {
      "RedBlueGreen "
      "#ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 "
      "#ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 "
      "#ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 "
      "#ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 "
      "#ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 "
      "#ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 "
      "#ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 "
      "#ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 #ff1010 "
      "#ff1010 #ff1010 #ff1010 #ff1010 #ff1010 "
      "#1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff "
      "#1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff "
      "#1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff "
      "#1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff "
      "#1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff "
      "#1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff "
      "#1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff "
      "#1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff #1010ff "
      "#1010ff #1010ff #1010ff #1010ff #1010ff "
      "#10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 "
      "#10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 "
      "#10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 "
      "#10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 "
      "#10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 "
      "#10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 "
      "#10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 "
      "#10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10 "
      "#10ff10 #10ff10 #10ff10 #10ff10 #10ff10 #10ff10"
};

/*
  [PT: May 28, 2019] Some new colormaps for ROIs, namely the
  GLASBEY colormaps, from here:

    https://github.com/taketwo/glasbey
    http://colorcet.pyviz.org/user_guide/Categorical.html

  For 256, 512, 1024 and 2048 colors.  For Ncolor > 256, one is
  assumed to be using the "-XXXnpane .." opt in afni.
*/
static char ROI_glasbey_256_CMD[] = {
"ROI_glasbey_256 "
"#00008f "
"#b60000 "
"#008c00 "
"#c34fff "
"#01a5ca "
"#ec9d00 "
"#76ff00 "
"#595354 "
"#ff7598 "
"#940073 "
"#00f3cc "
"#4853ff "
"#a6a19a "
"#004301 "
"#edb7ff "
"#8a6800 "
"#6100a3 "
"#5c0011 "
"#fff585 "
"#007b69 "
"#92b853 "
"#abd4ff "
"#7e79a3 "
"#ff5401 "
"#0a577d "
"#a8615c "
"#e700b9 "
"#ffc3a6 "
"#5b3500 "
"#00b485 "
"#7e9eff "
"#e7025c "
"#b8d8b7 "
"#c082b7 "
"#6f895b "
"#8a48a2 "
"#5b325a "
"#dc8a67 "
"#4f5c2c "
"#00e173 "
"#ff68ff "
"#7ec1c1 "
"#783a3d "
"#b7fcff "
"#8809ff "
"#6f8c90 "
"#aca8cc "
"#944305 "
"#00504a "
"#0100fa "
"#565194 "
"#cad800 "
"#097bc0 "
"#b0ff9b "
"#ad934b "
"#c46800 "
"#dacbd7 "
"#00ba00 "
"#ad5289 "
"#837772 "
"#9c003b "
"#906efe "
"#d5c373 "
"#feaccb "
"#00e9ff "
"#c192ff "
"#828e00 "
"#640041 "
"#576878 "
"#7d5e78 "
"#007134 "
"#7a5a3b "
"#352e77 "
"#d14839 "
"#5700e3 "
"#8ab28f "
"#be00cd "
"#821700 "
"#f4eac1 "
"#0f8395 "
"#ffc917 "
"#b8868c "
"#48c6ff "
"#596d5f "
"#c20076 "
"#fdff00 "
"#43485f "
"#7b99bd "
"#750084 "
"#587202 "
"#4ea34f "
"#c3af00 "
"#09a29d "
"#bdd4d9 "
"#abffd6 "
"#7ed200 "
"#d15a77 "
"#ff0022 "
"#ff5cc2 "
"#a17244 "
"#4980ff "
"#a06dbc "
"#d7d1ff "
"#a33b36 "
"#c8a5bc "
"#ff842b "
"#9b8da0 "
"#cbbfa6 "
"#d05ace "
"#0556b8 "
"#fe5c59 "
"#74d4a2 "
"#686bc4 "
"#843f66 "
"#00a1fc "
"#099363 "
"#ef94dd "
"#1e5532 "
"#b6d975 "
"#a800ef "
"#feab63 "
"#ff9f96 "
"#97a9b2 "
"#ff1695 "
"#006770 "
"#978ed4 "
"#695000 "
"#52ffaa "
"#aab8ff "
"#3c4300 "
"#55482d "
"#2f5b00 "
"#50006c "
"#ffd8ff "
"#632511 "
"#003b71 "
"#d2a398 "
"#639b84 "
"#00ceb4 "
"#674b7a "
"#97e0d1 "
"#a0728f "
"#8759ca "
"#dcffc7 "
"#e0ecff "
"#5e7ea1 "
"#adae7b "
"#948d6e "
"#726f3e "
"#8502be "
"#c3fc53 "
"#ffe2df "
"#425558 "
"#d5f2e5 "
"#00c3d7 "
"#dc8caa "
"#a5395d "
"#c10939 "
"#825759 "
"#3300bb "
"#bc4800 "
"#c76a47 "
"#9a00a2 "
"#446597 "
"#c68500 "
"#8db7d9 "
"#acbcb3 "
"#d0a46e "
"#d165a4 "
"#c99eda "
"#60c170 "
"#771934 "
"#fb00f4 "
"#74a905 "
"#503371 "
"#45853e "
"#674656 "
"#d37ee9 "
"#84eb64 "
"#b42498 "
"#8b9c51 "
"#777a85 "
"#59fff9 "
"#98deef "
"#ffd98a "
"#764021 "
"#dc797c "
"#6aa6b2 "
"#435241 "
"#7087ce "
"#6b39b5 "
"#9f8204 "
"#676286 "
"#d8d9d0 "
"#4e7778 "
"#b5886b "
"#ab52ad "
"#b7b3bb "
"#cab2fe "
"#7651fd "
"#b9c3dc "
"#41007e "
"#306354 "
"#3940a1 "
"#fa4b79 "
"#788679 "
"#a05d0d "
"#d91b00 "
"#ff7d5a "
"#00ea29 "
"#642c43 "
"#98c887 "
"#7a387c "
"#eabbbf "
"#e0d85c "
"#92503a "
"#dbe399 "
"#528369 "
"#d53e8c "
"#016f94 "
"#ceb9dd "
"#836398 "
"#db06fb "
"#5e025b "
"#4b7146 "
"#ab76eb "
"#899d95 "
"#4891c6 "
"#a6bf00 "
"#004b5b "
"#5a4bca "
"#9fe8a9 "
"#a688bc "
"#dd9041 "
"#605e48 "
"#74235e "
"#8b8342 "
"#5bb2a1 "
"#cca643 "
"#694940 "
"#6d666d "
"#7ab6fe "
"#a59c24 "
"#bf4253 "
"#227300 "
"#e36d27 "
"#46e1d9 "
"#9b556b "
"#9b0018 "
"#394d7d "
"#5c5467 "
"#8790a0 "
"#8f5087 "
"#8d2254 "
"#f5e9f8 "
"#fe83c1 "
};

static char ROI_glasbey_512_CMD[] = {
"ROI_glasbey_512 "
"#00008f "
"#b60000 "
"#008c00 "
"#c34fff "
"#01a5ca "
"#ec9d00 "
"#76ff00 "
"#595354 "
"#ff7598 "
"#940073 "
"#00f3cc "
"#4853ff "
"#a6a19a "
"#004301 "
"#edb7ff "
"#8a6800 "
"#6100a3 "
"#5c0011 "
"#fff585 "
"#007b69 "
"#92b853 "
"#abd4ff "
"#7e79a3 "
"#ff5401 "
"#0a577d "
"#a8615c "
"#e700b9 "
"#ffc3a6 "
"#5b3500 "
"#00b485 "
"#7e9eff "
"#e7025c "
"#b8d8b7 "
"#c082b7 "
"#6f895b "
"#8a48a2 "
"#5b325a "
"#dc8a67 "
"#4f5c2c "
"#00e173 "
"#ff68ff "
"#7ec1c1 "
"#783a3d "
"#b7fcff "
"#8809ff "
"#6f8c90 "
"#aca8cc "
"#944305 "
"#00504a "
"#0100fa "
"#565194 "
"#cad800 "
"#097bc0 "
"#b0ff9b "
"#ad934b "
"#c46800 "
"#dacbd7 "
"#00ba00 "
"#ad5289 "
"#837772 "
"#9c003b "
"#906efe "
"#d5c373 "
"#feaccb "
"#00e9ff "
"#c192ff "
"#828e00 "
"#640041 "
"#576878 "
"#7d5e78 "
"#007134 "
"#7a5a3b "
"#352e77 "
"#d14839 "
"#5700e3 "
"#8ab28f "
"#be00cd "
"#821700 "
"#f4eac1 "
"#0f8395 "
"#ffc917 "
"#b8868c "
"#48c6ff "
"#596d5f "
"#c20076 "
"#fdff00 "
"#43485f "
"#7b99bd "
"#750084 "
"#587202 "
"#4ea34f "
"#c3af00 "
"#09a29d "
"#bdd4d9 "
"#abffd6 "
"#7ed200 "
"#d15a77 "
"#ff0022 "
"#ff5cc2 "
"#a17244 "
"#4980ff "
"#a06dbc "
"#d7d1ff "
"#a33b36 "
"#c8a5bc "
"#ff842b "
"#9b8da0 "
"#cbbfa6 "
"#d05ace "
"#0556b8 "
"#fe5c59 "
"#74d4a2 "
"#686bc4 "
"#843f66 "
"#00a1fc "
"#099363 "
"#ef94dd "
"#1e5532 "
"#b6d975 "
"#a800ef "
"#feab63 "
"#ff9f96 "
"#97a9b2 "
"#ff1695 "
"#006770 "
"#978ed4 "
"#695000 "
"#52ffaa "
"#aab8ff "
"#3c4300 "
"#55482d "
"#2f5b00 "
"#50006c "
"#ffd8ff "
"#632511 "
"#003b71 "
"#d2a398 "
"#639b84 "
"#00ceb4 "
"#674b7a "
"#97e0d1 "
"#a0728f "
"#8759ca "
"#dcffc7 "
"#e0ecff "
"#5e7ea1 "
"#adae7b "
"#948d6e "
"#726f3e "
"#8502be "
"#c3fc53 "
"#ffe2df "
"#425558 "
"#d5f2e5 "
"#00c3d7 "
"#dc8caa "
"#a5395d "
"#c10939 "
"#825759 "
"#3300bb "
"#bc4800 "
"#c76a47 "
"#9a00a2 "
"#446597 "
"#c68500 "
"#8db7d9 "
"#acbcb3 "
"#d0a46e "
"#d165a4 "
"#c99eda "
"#60c170 "
"#771934 "
"#fb00f4 "
"#74a905 "
"#503371 "
"#45853e "
"#674656 "
"#d37ee9 "
"#84eb64 "
"#b42498 "
"#8b9c51 "
"#777a85 "
"#59fff9 "
"#98deef "
"#ffd98a "
"#764021 "
"#dc797c "
"#6aa6b2 "
"#435241 "
"#7087ce "
"#6b39b5 "
"#9f8204 "
"#676286 "
"#d8d9d0 "
"#4e7778 "
"#b5886b "
"#ab52ad "
"#b7b3bb "
"#cab2fe "
"#7651fd "
"#b9c3dc "
"#41007e "
"#306354 "
"#3940a1 "
"#fa4b79 "
"#788679 "
"#a05d0d "
"#d91b00 "
"#ff7d5a "
"#00ea29 "
"#642c43 "
"#98c887 "
"#7a387c "
"#eabbbf "
"#e0d85c "
"#92503a "
"#dbe399 "
"#528369 "
"#d53e8c "
"#016f94 "
"#ceb9dd "
"#836398 "
"#db06fb "
"#5e025b "
"#4b7146 "
"#ab76eb "
"#899d95 "
"#4891c6 "
"#a6bf00 "
"#004b5b "
"#5a4bca "
"#9fe8a9 "
"#a688bc "
"#dd9041 "
"#605e48 "
"#74235e "
"#8b8342 "
"#5bb2a1 "
"#cca643 "
"#694940 "
"#6d666d "
"#7ab6fe "
"#a59c24 "
"#bf4253 "
"#227300 "
"#e36d27 "
"#46e1d9 "
"#9b556b "
"#9b0018 "
"#394d7d "
"#5c5467 "
"#8790a0 "
"#8f5087 "
"#8d2254 "
"#f5e9f8 "
"#fe83c1 "
"#3639e8 "
"#6c358f "
"#4c09ac "
"#f1b84c "
"#a54cff "
"#b76d7f "
"#5a941b "
"#a0776f "
"#006feb "
"#9e2b00 "
"#00ff77 "
"#7a7813 "
"#5490a6 "
"#a79fff "
"#e474ce "
"#c17d41 "
"#8875c8 "
"#882929 "
"#50cb47 "
"#3f8c82 "
"#825112 "
"#ea98ff "
"#72a16d "
"#9b8a8a "
"#92c7b2 "
"#86748d "
"#b25238 "
"#666414 "
"#edbce2 "
"#d5f483 "
"#94a4db "
"#f6e109 "
"#e01235 "
"#72539f "
"#2c0089 "
"#bac054 "
"#c748a9 "
"#5fa8db "
"#bcc893 "
"#de4702 "
"#a4e701 "
"#7b84ff "
"#a84fd0 "
"#7b0017 "
"#ac7417 "
"#f3ffe6 "
"#867053 "
"#8c6b75 "
"#e357ff "
"#00cc8b "
"#00a422 "
"#75c9e3 "
"#8ad166 "
"#ff2351 "
"#e4bd8b "
"#ffcfe1 "
"#b369a4 "
"#4462c2 "
"#00379f "
"#a0c0cc "
"#eca683 "
"#feffb9 "
"#b49e7f "
"#4f3991 "
"#3e586d "
"#00e8a3 "
"#9f3476 "
"#51415f "
"#be766c "
"#ee579a "
"#00b94e "
"#6c8836 "
"#782d00 "
"#756259 "
"#52430b "
"#9693b7 "
"#dc5f5d "
"#95a184 "
"#217d54 "
"#ebd2b9 "
"#fd94ab "
"#660c00 "
"#c1ebff "
"#b9371f "
"#fd8382 "
"#00a083 "
"#92444d "
"#dbc50a "
"#b49aa1 "
"#b60b53 "
"#646fa0 "
"#07b8b6 "
"#82ebc0 "
"#dd7295 "
"#5b6263 "
"#e95c3a "
"#70765e "
"#cab4b3 "
"#e3a2ae "
"#4e78cc "
"#67bc94 "
"#008be7 "
"#b29ddf "
"#88b1a7 "
"#dd7b00 "
"#a028bf "
"#525300 "
"#d721d0 "
"#8a652e "
"#cc2e69 "
"#6e7472 "
"#47758a "
"#b29abb "
"#66272b "
"#8937ce "
"#898c8d "
"#916454 "
"#394d26 "
"#406b24 "
"#00963e "
"#d4d8b2 "
"#444070 "
"#6e13ff "
"#fe9664 "
"#696dff "
"#a19fa8 "
"#ea4bc9 "
"#a086fe "
"#dcfaff "
"#b38ba4 "
"#bd6ad8 "
"#a8ac56 "
"#004c8f "
"#675b2c "
"#5e256f "
"#274764 "
"#de4157 "
"#91af71 "
"#a7d644 "
"#b64f71 "
"#7effe2 "
"#0987b1 "
"#df918b "
"#68481c "
"#69b449 "
"#77dd8a "
"#d496c3 "
"#c37b9d "
"#e3b000 "
"#744a6e "
"#d2d5e4 "
"#ffadf1 "
"#a60065 "
"#8d3b23 "
"#505778 "
"#607534 "
"#64372b "
"#9d3b99 "
"#46a36e "
"#c5c7c9 "
"#7668a9 "
"#5fb4d1 "
"#006743 "
"#e3e9eb "
"#620029 "
"#99a900 "
"#a35f37 "
"#f1ff68 "
"#e1745a "
"#c3eea4 "
"#c20afb "
"#adb19a "
"#5a93ee "
"#0100bc "
"#ff26b9 "
"#3f6041 "
"#609c98 "
"#9a76a3 "
"#19297e "
"#035518 "
"#cf0199 "
"#96a6c2 "
"#cbdcff "
"#bdbbeb "
"#a6c2a1 "
"#009aaa "
"#b886db "
"#00797b "
"#c09705 "
"#0068ad "
"#822599 "
"#00d5fb "
"#c4f0cb "
"#6400c9 "
"#7e0052 "
"#81be2b "
"#74a03d "
"#ef413e "
"#c872ff "
"#42656c "
"#657d8b "
"#53905e "
"#7887b0 "
"#baded2 "
"#7b967a "
"#67dabd "
"#3f559f "
"#ff9a30 "
"#c05955 "
"#744ad0 "
"#cd5c1f "
"#e1b8a4 "
"#86c9ff "
"#e6007c "
"#00b4fc "
"#6d575f "
"#ffc275 "
"#15cdcb "
"#c045d3 "
"#861980 "
"#b5f4e4 "
"#a38555 "
"#942d43 "
"#82f490 "
"#e2ceee "
"#4b4d55 "
"#72009d "
"#0055ec "
"#2b5142 "
"#ffd75f "
"#a0cdca "
"#dbef09 "
"#9c9b66 "
"#99629c "
"#897f88 "
"#825167 "
"#8fecec "
"#69728b "
"#b2ec6c "
"#f16878 "
"#00d300 "
"#f83400 "
"#355c27 "
"#f1ede2 "
"#4b8203 "
"#df7bb3 "
"#25575d "
"#7b324c "
"#41725b "
"#e1a8de "
"#955f80 "
"#ba3885 "
"#63837b "
"#baffbf "
"#8e45fe "
"#c39154 "
};

static char ROI_glasbey_1024_CMD[] = {
"ROI_glasbey_1024 "
"#00008f "
"#b60000 "
"#008c00 "
"#c34fff "
"#01a5ca "
"#ec9d00 "
"#76ff00 "
"#595354 "
"#ff7598 "
"#940073 "
"#00f3cc "
"#4853ff "
"#a6a19a "
"#004301 "
"#edb7ff "
"#8a6800 "
"#6100a3 "
"#5c0011 "
"#fff585 "
"#007b69 "
"#92b853 "
"#abd4ff "
"#7e79a3 "
"#ff5401 "
"#0a577d "
"#a8615c "
"#e700b9 "
"#ffc3a6 "
"#5b3500 "
"#00b485 "
"#7e9eff "
"#e7025c "
"#b8d8b7 "
"#c082b7 "
"#6f895b "
"#8a48a2 "
"#5b325a "
"#dc8a67 "
"#4f5c2c "
"#00e173 "
"#ff68ff "
"#7ec1c1 "
"#783a3d "
"#b7fcff "
"#8809ff "
"#6f8c90 "
"#aca8cc "
"#944305 "
"#00504a "
"#0100fa "
"#565194 "
"#cad800 "
"#097bc0 "
"#b0ff9b "
"#ad934b "
"#c46800 "
"#dacbd7 "
"#00ba00 "
"#ad5289 "
"#837772 "
"#9c003b "
"#906efe "
"#d5c373 "
"#feaccb "
"#00e9ff "
"#c192ff "
"#828e00 "
"#640041 "
"#576878 "
"#7d5e78 "
"#007134 "
"#7a5a3b "
"#352e77 "
"#d14839 "
"#5700e3 "
"#8ab28f "
"#be00cd "
"#821700 "
"#f4eac1 "
"#0f8395 "
"#ffc917 "
"#b8868c "
"#48c6ff "
"#596d5f "
"#c20076 "
"#fdff00 "
"#43485f "
"#7b99bd "
"#750084 "
"#587202 "
"#4ea34f "
"#c3af00 "
"#09a29d "
"#bdd4d9 "
"#abffd6 "
"#7ed200 "
"#d15a77 "
"#ff0022 "
"#ff5cc2 "
"#a17244 "
"#4980ff "
"#a06dbc "
"#d7d1ff "
"#a33b36 "
"#c8a5bc "
"#ff842b "
"#9b8da0 "
"#cbbfa6 "
"#d05ace "
"#0556b8 "
"#fe5c59 "
"#74d4a2 "
"#686bc4 "
"#843f66 "
"#00a1fc "
"#099363 "
"#ef94dd "
"#1e5532 "
"#b6d975 "
"#a800ef "
"#feab63 "
"#ff9f96 "
"#97a9b2 "
"#ff1695 "
"#006770 "
"#978ed4 "
"#695000 "
"#52ffaa "
"#aab8ff "
"#3c4300 "
"#55482d "
"#2f5b00 "
"#50006c "
"#ffd8ff "
"#632511 "
"#003b71 "
"#d2a398 "
"#639b84 "
"#00ceb4 "
"#674b7a "
"#97e0d1 "
"#a0728f "
"#8759ca "
"#dcffc7 "
"#e0ecff "
"#5e7ea1 "
"#adae7b "
"#948d6e "
"#726f3e "
"#8502be "
"#c3fc53 "
"#ffe2df "
"#425558 "
"#d5f2e5 "
"#00c3d7 "
"#dc8caa "
"#a5395d "
"#c10939 "
"#825759 "
"#3300bb "
"#bc4800 "
"#c76a47 "
"#9a00a2 "
"#446597 "
"#c68500 "
"#8db7d9 "
"#acbcb3 "
"#d0a46e "
"#d165a4 "
"#c99eda "
"#60c170 "
"#771934 "
"#fb00f4 "
"#74a905 "
"#503371 "
"#45853e "
"#674656 "
"#d37ee9 "
"#84eb64 "
"#b42498 "
"#8b9c51 "
"#777a85 "
"#59fff9 "
"#98deef "
"#ffd98a "
"#764021 "
"#dc797c "
"#6aa6b2 "
"#435241 "
"#7087ce "
"#6b39b5 "
"#9f8204 "
"#676286 "
"#d8d9d0 "
"#4e7778 "
"#b5886b "
"#ab52ad "
"#b7b3bb "
"#cab2fe "
"#7651fd "
"#b9c3dc "
"#41007e "
"#306354 "
"#3940a1 "
"#fa4b79 "
"#788679 "
"#a05d0d "
"#d91b00 "
"#ff7d5a "
"#00ea29 "
"#642c43 "
"#98c887 "
"#7a387c "
"#eabbbf "
"#e0d85c "
"#92503a "
"#dbe399 "
"#528369 "
"#d53e8c "
"#016f94 "
"#ceb9dd "
"#836398 "
"#db06fb "
"#5e025b "
"#4b7146 "
"#ab76eb "
"#899d95 "
"#4891c6 "
"#a6bf00 "
"#004b5b "
"#5a4bca "
"#9fe8a9 "
"#a688bc "
"#dd9041 "
"#605e48 "
"#74235e "
"#8b8342 "
"#5bb2a1 "
"#cca643 "
"#694940 "
"#6d666d "
"#7ab6fe "
"#a59c24 "
"#bf4253 "
"#227300 "
"#e36d27 "
"#46e1d9 "
"#9b556b "
"#9b0018 "
"#394d7d "
"#5c5467 "
"#8790a0 "
"#8f5087 "
"#8d2254 "
"#f5e9f8 "
"#fe83c1 "
"#3639e8 "
"#6c358f "
"#4c09ac "
"#f1b84c "
"#a54cff "
"#b76d7f "
"#5a941b "
"#a0776f "
"#006feb "
"#9e2b00 "
"#00ff77 "
"#7a7813 "
"#5490a6 "
"#a79fff "
"#e474ce "
"#c17d41 "
"#8875c8 "
"#882929 "
"#50cb47 "
"#3f8c82 "
"#825112 "
"#ea98ff "
"#72a16d "
"#9b8a8a "
"#92c7b2 "
"#86748d "
"#b25238 "
"#666414 "
"#edbce2 "
"#d5f483 "
"#94a4db "
"#f6e109 "
"#e01235 "
"#72539f "
"#2c0089 "
"#bac054 "
"#c748a9 "
"#5fa8db "
"#bcc893 "
"#de4702 "
"#a4e701 "
"#7b84ff "
"#a84fd0 "
"#7b0017 "
"#ac7417 "
"#f3ffe6 "
"#867053 "
"#8c6b75 "
"#e357ff "
"#00cc8b "
"#00a422 "
"#75c9e3 "
"#8ad166 "
"#ff2351 "
"#e4bd8b "
"#ffcfe1 "
"#b369a4 "
"#4462c2 "
"#00379f "
"#a0c0cc "
"#eca683 "
"#feffb9 "
"#b49e7f "
"#4f3991 "
"#3e586d "
"#00e8a3 "
"#9f3476 "
"#51415f "
"#be766c "
"#ee579a "
"#00b94e "
"#6c8836 "
"#782d00 "
"#756259 "
"#52430b "
"#9693b7 "
"#dc5f5d "
"#95a184 "
"#217d54 "
"#ebd2b9 "
"#fd94ab "
"#660c00 "
"#c1ebff "
"#b9371f "
"#fd8382 "
"#00a083 "
"#92444d "
"#dbc50a "
"#b49aa1 "
"#b60b53 "
"#646fa0 "
"#07b8b6 "
"#82ebc0 "
"#dd7295 "
"#5b6263 "
"#e95c3a "
"#70765e "
"#cab4b3 "
"#e3a2ae "
"#4e78cc "
"#67bc94 "
"#008be7 "
"#b29ddf "
"#88b1a7 "
"#dd7b00 "
"#a028bf "
"#525300 "
"#d721d0 "
"#8a652e "
"#cc2e69 "
"#6e7472 "
"#47758a "
"#b29abb "
"#66272b "
"#8937ce "
"#898c8d "
"#916454 "
"#394d26 "
"#406b24 "
"#00963e "
"#d4d8b2 "
"#444070 "
"#6e13ff "
"#fe9664 "
"#696dff "
"#a19fa8 "
"#ea4bc9 "
"#a086fe "
"#dcfaff "
"#b38ba4 "
"#bd6ad8 "
"#a8ac56 "
"#004c8f "
"#675b2c "
"#5e256f "
"#274764 "
"#de4157 "
"#91af71 "
"#a7d644 "
"#b64f71 "
"#7effe2 "
"#0987b1 "
"#df918b "
"#68481c "
"#69b449 "
"#77dd8a "
"#d496c3 "
"#c37b9d "
"#e3b000 "
"#744a6e "
"#d2d5e4 "
"#ffadf1 "
"#a60065 "
"#8d3b23 "
"#505778 "
"#607534 "
"#64372b "
"#9d3b99 "
"#46a36e "
"#c5c7c9 "
"#7668a9 "
"#5fb4d1 "
"#006743 "
"#e3e9eb "
"#620029 "
"#99a900 "
"#a35f37 "
"#f1ff68 "
"#e1745a "
"#c3eea4 "
"#c20afb "
"#adb19a "
"#5a93ee "
"#0100bc "
"#ff26b9 "
"#3f6041 "
"#609c98 "
"#9a76a3 "
"#19297e "
"#035518 "
"#cf0199 "
"#96a6c2 "
"#cbdcff "
"#bdbbeb "
"#a6c2a1 "
"#009aaa "
"#b886db "
"#00797b "
"#c09705 "
"#0068ad "
"#822599 "
"#00d5fb "
"#c4f0cb "
"#6400c9 "
"#7e0052 "
"#81be2b "
"#74a03d "
"#ef413e "
"#c872ff "
"#42656c "
"#657d8b "
"#53905e "
"#7887b0 "
"#baded2 "
"#7b967a "
"#67dabd "
"#3f559f "
"#ff9a30 "
"#c05955 "
"#744ad0 "
"#cd5c1f "
"#e1b8a4 "
"#86c9ff "
"#e6007c "
"#00b4fc "
"#6d575f "
"#ffc275 "
"#15cdcb "
"#c045d3 "
"#861980 "
"#b5f4e4 "
"#a38555 "
"#942d43 "
"#82f490 "
"#e2ceee "
"#4b4d55 "
"#72009d "
"#0055ec "
"#2b5142 "
"#ffd75f "
"#a0cdca "
"#dbef09 "
"#9c9b66 "
"#99629c "
"#897f88 "
"#825167 "
"#8fecec "
"#69728b "
"#b2ec6c "
"#f16878 "
"#00d300 "
"#f83400 "
"#355c27 "
"#f1ede2 "
"#4b8203 "
"#df7bb3 "
"#25575d "
"#7b324c "
"#41725b "
"#e1a8de "
"#955f80 "
"#ba3885 "
"#63837b "
"#baffbf "
"#8e45fe "
"#c39154 "
"#769ddc "
"#e3a653 "
"#6d395a "
"#7797a7 "
"#ae2126 "
"#a6bfeb "
"#4e878e "
"#c1cbb9 "
"#ff89fb "
"#95ff5c "
"#a94e58 "
"#ff0e74 "
"#773122 "
"#e3d196 "
"#5f458a "
"#95752e "
"#3a00e5 "
"#d7be48 "
"#b1d99a "
"#11ba70 "
"#ad2d45 "
"#ac6e55 "
"#ef99c6 "
"#abbe74 "
"#e24077 "
"#554854 "
"#bb9489 "
"#daeacf "
"#76d4db "
"#00802c "
"#918e30 "
"#9f7d88 "
"#723e00 "
"#c8ba87 "
"#675ab0 "
"#915bb5 "
"#b7e6e8 "
"#ffb0b6 "
"#927ab4 "
"#dcc8c4 "
"#add1e6 "
"#9d64d6 "
"#c81a21 "
"#ff6f2f "
"#5e6b4a "
"#dab8cb "
"#ffcdc7 "
"#d58ad3 "
"#633a40 "
"#2b3d7f "
"#ff7ddc "
"#868299 "
"#9bdab6 "
"#d4a1fd "
"#a4411e "
"#c995a3 "
"#e6e1fe "
"#7e65dc "
"#fed3a3 "
"#501a7c "
"#be608a "
"#b45b1b "
"#7e4a40 "
"#84d9ff "
"#495e55 "
"#33aebf "
"#d2e654 "
"#82b5c4 "
"#3f75a1 "
"#80825b "
"#cbad8e "
"#e88246 "
"#cd9879 "
"#2a4b00 "
"#629fbd "
"#73677e "
"#0540c3 "
"#7f00da "
"#ff5be1 "
"#4e6100 "
"#c975ca "
"#7c5287 "
"#6dc5b2 "
"#ffb000 "
"#006b63 "
"#bba561 "
"#701d1a "
"#3d6683 "
"#aab2c4 "
"#929488 "
"#987e65 "
"#dbc1fb "
"#e449a9 "
"#9faaa7 "
"#7979db "
"#785f19 "
"#7fa1a3 "
"#008c71 "
"#56ffc7 "
"#ea099c "
"#9de386 "
"#26ff47 "
"#8d3a7c "
"#5fa6fd "
"#00cc64 "
"#76b47a "
"#607b5a "
"#5738b8 "
"#5f1d4e "
"#679650 "
"#6a8100 "
"#e5ffa9 "
"#7579b8 "
"#bbacc4 "
"#018e91 "
"#006715 "
"#5633ff "
"#beccff "
"#8f92f1 "
"#c4d14a "
"#75006c "
"#adff00 "
"#974b76 "
"#b17abd "
"#67685f "
"#c6c1d2 "
"#525968 "
"#0ad8a8 "
"#c9696f "
"#554b74 "
"#9c1cff "
"#ffc5ff "
"#504e42 "
"#a06772 "
"#e7dee1 "
"#5f3a6f "
"#701d45 "
"#b7b5b0 "
"#c98878 "
"#56df55 "
"#5662a6 "
"#cd8190 "
"#f38f78 "
"#c8d180 "
"#642c00 "
"#5e5dde "
"#8693ca "
"#6caa85 "
"#88003f "
"#7b40a8 "
"#8cc59b "
"#9d1e8b "
"#ff6bad "
"#eb54e7 "
"#78e72c "
"#cb05b3 "
"#afbbc0 "
"#e2a9c9 "
"#a5907e "
"#92fbba "
"#275a92 "
"#b56c39 "
"#b78535 "
"#8d5931 "
"#849666 "
"#035b6f "
"#586688 "
"#4e4e25 "
"#d0284f "
"#794353 "
"#5ac200 "
"#00bae8 "
"#eb8496 "
"#acabe6 "
"#472685 "
"#ecaba1 "
"#ab0080 "
"#c1ace0 "
"#324d5a "
"#91a63d "
"#d49627 "
"#56a62e "
"#5e6f73 "
"#008949 "
"#ffe857 "
"#e6d67c "
"#127385 "
"#15a0dc "
"#466bff "
"#005b45 "
"#e89c62 "
"#768298 "
"#81a6bd "
"#bca599 "
"#c57c5b "
"#486b64 "
"#955553 "
"#5f3c1f "
"#b864bc "
"#f8eb9c "
"#26d6e5 "
"#5883bc "
"#67533d "
"#6e266e "
"#ff00d7 "
"#4225b8 "
"#747e44 "
"#abb136 "
"#a081d0 "
"#b000ae "
"#882a67 "
"#84816e "
"#614963 "
"#cbe5b5 "
"#7aac5d "
"#6d5877 "
"#c2b45d "
"#00bc9e "
"#72eba6 "
"#ec8c07 "
"#736646 "
"#be3bb8 "
"#8f3a58 "
"#89132d "
"#ca3801 "
"#749187 "
"#6fecd8 "
"#636474 "
"#ff4c31 "
"#701faf "
"#91adfc "
"#5eb9c2 "
"#99271d "
"#9a46db "
"#d0e6ee "
"#909a9f "
"#02f38c "
"#404fc2 "
"#a4c857 "
"#7aedff "
"#4c5250 "
"#ed7eff "
"#ffb686 "
"#8f7a4b "
"#5c4649 "
"#479273 "
"#8a86b9 "
"#7f7128 "
"#bc8dc9 "
"#447d70 "
"#4a8053 "
"#6d8ca8 "
"#b0469b "
"#afcdbd "
"#8c688c "
"#bd3438 "
"#758183 "
"#00a358 "
"#f0d7eb "
"#a485a7 "
"#635589 "
"#3e6fb3 "
"#70b8e9 "
"#e5f0b9 "
"#edfff9 "
"#99ad9a "
"#73b0af "
"#5b2f8d "
"#03622d "
"#83acdf "
"#3f5213 "
"#b65d6a "
"#dcffe1 "
"#d8b36c "
"#61d26c "
"#9e1f31 "
"#cca6ad "
"#9cbbb9 "
"#620082 "
"#7e707b "
"#a18435 "
"#c967b6 "
"#d58cff "
"#a96584 "
"#7ed2c8 "
"#8c3200 "
"#214b1d "
"#84be6e "
"#2f2e94 "
"#7acb89 "
"#4ca98b "
"#f57267 "
"#ab8ee8 "
"#ff6747 "
"#884922 "
"#d5588e "
"#a01f52 "
"#40d98a "
"#cc6f31 "
"#bfc022 "
"#454689 "
"#407922 "
"#7e8d4b "
"#a3501e "
"#e13a22 "
"#94c9e9 "
"#20019f "
"#a00e00 "
"#f3bcd2 "
"#f44a5e "
"#0965cb "
"#b4a042 "
"#77438c "
"#ac9028 "
"#cca9d0 "
"#2195b6 "
"#a9ebcb "
"#ffe3ca "
"#fff2f5 "
"#9483a8 "
"#cedcd9 "
"#876b65 "
"#c3791c "
"#76a397 "
"#a24d43 "
"#796c92 "
"#5f6833 "
"#8e82e1 "
"#99fff5 "
"#896ab1 "
"#4c933c "
"#31693c "
"#6a896e "
"#8d5c03 "
"#56af69 "
"#48829f "
"#802c3a "
"#e85f85 "
"#cd51ec "
"#9fb38c "
"#51604a "
"#b7316c "
"#00f1f3 "
"#893b37 "
"#671423 "
"#a39db5 "
"#e5e87f "
"#056e57 "
"#d6aeee "
"#9806d1 "
"#3f5a80 "
"#191bca "
"#9f9cd0 "
"#f67100 "
"#c75839 "
"#e563ba "
"#3c108e "
"#4ea9aa "
"#c1ff7f "
"#efd1d6 "
"#b09464 "
"#78564a "
"#967e93 "
"#ad79a4 "
"#b230fe "
"#734c33 "
"#b230d4 "
"#38783d "
"#f3f5ff "
"#677a6c "
"#a367fe "
"#bca3ff "
"#17ac98 "
"#ef7ea8 "
"#c34e92 "
"#69dbf3 "
"#94d69d "
"#cffff7 "
"#79ff7a "
"#9f48b5 "
"#4539c7 "
"#a8e0ff "
"#b07a8f "
"#539aa4 "
"#701bde "
"#b0a0b0 "
"#a94474 "
"#aa867f "
"#95cdd9 "
"#6b90c4 "
"#e85a00 "
"#8e7711 "
"#b6a9ac "
"#315853 "
"#fea0ff "
"#cac5e6 "
"#e7ad74 "
"#7a6469 "
"#c5d9eb "
"#5e8247 "
"#31416a "
"#86da49 "
"#849e16 "
"#a3b0d8 "
"#b6624d "
"#6b3c6a "
"#4c4aa8 "
"#637ee1 "
"#e371e6 "
"#8c308f "
"#eec76d "
"#5d5b62 "
"#a6f0ff "
"#933fbf "
"#83a48d "
"#685550 "
"#fffed3 "
"#712006 "
"#b54843 "
"#714c51 "
"#5a205b "
"#813c06 "
"#75003b "
"#8b8200 "
"#00ffe6 "
"#c88cad "
"#004975 "
"#97beff "
"#a4bbd5 "
"#ff5d93 "
"#529ce0 "
"#595c55 "
"#786f5f "
"#745524 "
"#af7679 "
"#5fea77 "
"#7e9336 "
"#a58c98 "
"#ffac40 "
"#999597 "
"#916845 "
"#6f2982 "
"#015d59 "
"#9400b5 "
"#5c3c54 "
"#a997c5 "
"#cabbc5 "
"#eecbff "
"#cb4878 "
"#842b14 "
"#9c6805 "
"#b9c0a0 "
"#d5a10a "
"#4d5a8e "
"#6b6e75 "
"#646199 "
"#e039e6 "
"#99dbdc "
"#d60986 "
"#b27b53 "
"#45c7a1 "
"#ffaf96 "
"#651e34 "
"#8d5d6b "
"#59bdb6 "
"#78c353 "
"#ae0239 "
"#6f0356 "
"#7e7d7e "
"#786900 "
"#925eea "
"#b9e540 "
"#be92b9 "
"#51543a "
"#604aab "
"#828c71 "
"#547190 "
"#c9dcca "
"#efc749 "
"#f39246 "
"#b265e5 "
"#bbb6d4 "
"#d593e8 "
"#c29742 "
"#96519d "
"#7a356a "
"#d28851 "
"#a25695 "
"#9f9947 "
"#002fff "
"#5ab423 "
"#891517 "
"#88a57a "
"#9e4487 "
"#dfd72e "
"#00e2c7 "
"#356f6e "
"#008acb "
"#725a8a "
"#7790ea "
"#d635a2 "
"#b03800 "
"#c297e7 "
"#5b5217 "
"#669772 "
"#ce6e5f "
"#f03465 "
"#ff9fdd "
"#e9001b "
"#6e5368 "
"#ab34b4 "
"#c8c4bc "
};


static char ROI_glasbey_2048_CMD[] = {
"ROI_glasbey_2048 "
"#00008f "
"#b60000 "
"#008c00 "
"#c34fff "
"#01a5ca "
"#ec9d00 "
"#76ff00 "
"#595354 "
"#ff7598 "
"#940073 "
"#00f3cc "
"#4853ff "
"#a6a19a "
"#004301 "
"#edb7ff "
"#8a6800 "
"#6100a3 "
"#5c0011 "
"#fff585 "
"#007b69 "
"#92b853 "
"#abd4ff "
"#7e79a3 "
"#ff5401 "
"#0a577d "
"#a8615c "
"#e700b9 "
"#ffc3a6 "
"#5b3500 "
"#00b485 "
"#7e9eff "
"#e7025c "
"#b8d8b7 "
"#c082b7 "
"#6f895b "
"#8a48a2 "
"#5b325a "
"#dc8a67 "
"#4f5c2c "
"#00e173 "
"#ff68ff "
"#7ec1c1 "
"#783a3d "
"#b7fcff "
"#8809ff "
"#6f8c90 "
"#aca8cc "
"#944305 "
"#00504a "
"#0100fa "
"#565194 "
"#cad800 "
"#097bc0 "
"#b0ff9b "
"#ad934b "
"#c46800 "
"#dacbd7 "
"#00ba00 "
"#ad5289 "
"#837772 "
"#9c003b "
"#906efe "
"#d5c373 "
"#feaccb "
"#00e9ff "
"#c192ff "
"#828e00 "
"#640041 "
"#576878 "
"#7d5e78 "
"#007134 "
"#7a5a3b "
"#352e77 "
"#d14839 "
"#5700e3 "
"#8ab28f "
"#be00cd "
"#821700 "
"#f4eac1 "
"#0f8395 "
"#ffc917 "
"#b8868c "
"#48c6ff "
"#596d5f "
"#c20076 "
"#fdff00 "
"#43485f "
"#7b99bd "
"#750084 "
"#587202 "
"#4ea34f "
"#c3af00 "
"#09a29d "
"#bdd4d9 "
"#abffd6 "
"#7ed200 "
"#d15a77 "
"#ff0022 "
"#ff5cc2 "
"#a17244 "
"#4980ff "
"#a06dbc "
"#d7d1ff "
"#a33b36 "
"#c8a5bc "
"#ff842b "
"#9b8da0 "
"#cbbfa6 "
"#d05ace "
"#0556b8 "
"#fe5c59 "
"#74d4a2 "
"#686bc4 "
"#843f66 "
"#00a1fc "
"#099363 "
"#ef94dd "
"#1e5532 "
"#b6d975 "
"#a800ef "
"#feab63 "
"#ff9f96 "
"#97a9b2 "
"#ff1695 "
"#006770 "
"#978ed4 "
"#695000 "
"#52ffaa "
"#aab8ff "
"#3c4300 "
"#55482d "
"#2f5b00 "
"#50006c "
"#ffd8ff "
"#632511 "
"#003b71 "
"#d2a398 "
"#639b84 "
"#00ceb4 "
"#674b7a "
"#97e0d1 "
"#a0728f "
"#8759ca "
"#dcffc7 "
"#e0ecff "
"#5e7ea1 "
"#adae7b "
"#948d6e "
"#726f3e "
"#8502be "
"#c3fc53 "
"#ffe2df "
"#425558 "
"#d5f2e5 "
"#00c3d7 "
"#dc8caa "
"#a5395d "
"#c10939 "
"#825759 "
"#3300bb "
"#bc4800 "
"#c76a47 "
"#9a00a2 "
"#446597 "
"#c68500 "
"#8db7d9 "
"#acbcb3 "
"#d0a46e "
"#d165a4 "
"#c99eda "
"#60c170 "
"#771934 "
"#fb00f4 "
"#74a905 "
"#503371 "
"#45853e "
"#674656 "
"#d37ee9 "
"#84eb64 "
"#b42498 "
"#8b9c51 "
"#777a85 "
"#59fff9 "
"#98deef "
"#ffd98a "
"#764021 "
"#dc797c "
"#6aa6b2 "
"#435241 "
"#7087ce "
"#6b39b5 "
"#9f8204 "
"#676286 "
"#d8d9d0 "
"#4e7778 "
"#b5886b "
"#ab52ad "
"#b7b3bb "
"#cab2fe "
"#7651fd "
"#b9c3dc "
"#41007e "
"#306354 "
"#3940a1 "
"#fa4b79 "
"#788679 "
"#a05d0d "
"#d91b00 "
"#ff7d5a "
"#00ea29 "
"#642c43 "
"#98c887 "
"#7a387c "
"#eabbbf "
"#e0d85c "
"#92503a "
"#dbe399 "
"#528369 "
"#d53e8c "
"#016f94 "
"#ceb9dd "
"#836398 "
"#db06fb "
"#5e025b "
"#4b7146 "
"#ab76eb "
"#899d95 "
"#4891c6 "
"#a6bf00 "
"#004b5b "
"#5a4bca "
"#9fe8a9 "
"#a688bc "
"#dd9041 "
"#605e48 "
"#74235e "
"#8b8342 "
"#5bb2a1 "
"#cca643 "
"#694940 "
"#6d666d "
"#7ab6fe "
"#a59c24 "
"#bf4253 "
"#227300 "
"#e36d27 "
"#46e1d9 "
"#9b556b "
"#9b0018 "
"#394d7d "
"#5c5467 "
"#8790a0 "
"#8f5087 "
"#8d2254 "
"#f5e9f8 "
"#fe83c1 "
"#3639e8 "
"#6c358f "
"#4c09ac "
"#f1b84c "
"#a54cff "
"#b76d7f "
"#5a941b "
"#a0776f "
"#006feb "
"#9e2b00 "
"#00ff77 "
"#7a7813 "
"#5490a6 "
"#a79fff "
"#e474ce "
"#c17d41 "
"#8875c8 "
"#882929 "
"#50cb47 "
"#3f8c82 "
"#825112 "
"#ea98ff "
"#72a16d "
"#9b8a8a "
"#92c7b2 "
"#86748d "
"#b25238 "
"#666414 "
"#edbce2 "
"#d5f483 "
"#94a4db "
"#f6e109 "
"#e01235 "
"#72539f "
"#2c0089 "
"#bac054 "
"#c748a9 "
"#5fa8db "
"#bcc893 "
"#de4702 "
"#a4e701 "
"#7b84ff "
"#a84fd0 "
"#7b0017 "
"#ac7417 "
"#f3ffe6 "
"#867053 "
"#8c6b75 "
"#e357ff "
"#00cc8b "
"#00a422 "
"#75c9e3 "
"#8ad166 "
"#ff2351 "
"#e4bd8b "
"#ffcfe1 "
"#b369a4 "
"#4462c2 "
"#00379f "
"#a0c0cc "
"#eca683 "
"#feffb9 "
"#b49e7f "
"#4f3991 "
"#3e586d "
"#00e8a3 "
"#9f3476 "
"#51415f "
"#be766c "
"#ee579a "
"#00b94e "
"#6c8836 "
"#782d00 "
"#756259 "
"#52430b "
"#9693b7 "
"#dc5f5d "
"#95a184 "
"#217d54 "
"#ebd2b9 "
"#fd94ab "
"#660c00 "
"#c1ebff "
"#b9371f "
"#fd8382 "
"#00a083 "
"#92444d "
"#dbc50a "
"#b49aa1 "
"#b60b53 "
"#646fa0 "
"#07b8b6 "
"#82ebc0 "
"#dd7295 "
"#5b6263 "
"#e95c3a "
"#70765e "
"#cab4b3 "
"#e3a2ae "
"#4e78cc "
"#67bc94 "
"#008be7 "
"#b29ddf "
"#88b1a7 "
"#dd7b00 "
"#a028bf "
"#525300 "
"#d721d0 "
"#8a652e "
"#cc2e69 "
"#6e7472 "
"#47758a "
"#b29abb "
"#66272b "
"#8937ce "
"#898c8d "
"#916454 "
"#394d26 "
"#406b24 "
"#00963e "
"#d4d8b2 "
"#444070 "
"#6e13ff "
"#fe9664 "
"#696dff "
"#a19fa8 "
"#ea4bc9 "
"#a086fe "
"#dcfaff "
"#b38ba4 "
"#bd6ad8 "
"#a8ac56 "
"#004c8f "
"#675b2c "
"#5e256f "
"#274764 "
"#de4157 "
"#91af71 "
"#a7d644 "
"#b64f71 "
"#7effe2 "
"#0987b1 "
"#df918b "
"#68481c "
"#69b449 "
"#77dd8a "
"#d496c3 "
"#c37b9d "
"#e3b000 "
"#744a6e "
"#d2d5e4 "
"#ffadf1 "
"#a60065 "
"#8d3b23 "
"#505778 "
"#607534 "
"#64372b "
"#9d3b99 "
"#46a36e "
"#c5c7c9 "
"#7668a9 "
"#5fb4d1 "
"#006743 "
"#e3e9eb "
"#620029 "
"#99a900 "
"#a35f37 "
"#f1ff68 "
"#e1745a "
"#c3eea4 "
"#c20afb "
"#adb19a "
"#5a93ee "
"#0100bc "
"#ff26b9 "
"#3f6041 "
"#609c98 "
"#9a76a3 "
"#19297e "
"#035518 "
"#cf0199 "
"#96a6c2 "
"#cbdcff "
"#bdbbeb "
"#a6c2a1 "
"#009aaa "
"#b886db "
"#00797b "
"#c09705 "
"#0068ad "
"#822599 "
"#00d5fb "
"#c4f0cb "
"#6400c9 "
"#7e0052 "
"#81be2b "
"#74a03d "
"#ef413e "
"#c872ff "
"#42656c "
"#657d8b "
"#53905e "
"#7887b0 "
"#baded2 "
"#7b967a "
"#67dabd "
"#3f559f "
"#ff9a30 "
"#c05955 "
"#744ad0 "
"#cd5c1f "
"#e1b8a4 "
"#86c9ff "
"#e6007c "
"#00b4fc "
"#6d575f "
"#ffc275 "
"#15cdcb "
"#c045d3 "
"#861980 "
"#b5f4e4 "
"#a38555 "
"#942d43 "
"#82f490 "
"#e2ceee "
"#4b4d55 "
"#72009d "
"#0055ec "
"#2b5142 "
"#ffd75f "
"#a0cdca "
"#dbef09 "
"#9c9b66 "
"#99629c "
"#897f88 "
"#825167 "
"#8fecec "
"#69728b "
"#b2ec6c "
"#f16878 "
"#00d300 "
"#f83400 "
"#355c27 "
"#f1ede2 "
"#4b8203 "
"#df7bb3 "
"#25575d "
"#7b324c "
"#41725b "
"#e1a8de "
"#955f80 "
"#ba3885 "
"#63837b "
"#baffbf "
"#8e45fe "
"#c39154 "
"#769ddc "
"#e3a653 "
"#6d395a "
"#7797a7 "
"#ae2126 "
"#a6bfeb "
"#4e878e "
"#c1cbb9 "
"#ff89fb "
"#95ff5c "
"#a94e58 "
"#ff0e74 "
"#773122 "
"#e3d196 "
"#5f458a "
"#95752e "
"#3a00e5 "
"#d7be48 "
"#b1d99a "
"#11ba70 "
"#ad2d45 "
"#ac6e55 "
"#ef99c6 "
"#abbe74 "
"#e24077 "
"#554854 "
"#bb9489 "
"#daeacf "
"#76d4db "
"#00802c "
"#918e30 "
"#9f7d88 "
"#723e00 "
"#c8ba87 "
"#675ab0 "
"#915bb5 "
"#b7e6e8 "
"#ffb0b6 "
"#927ab4 "
"#dcc8c4 "
"#add1e6 "
"#9d64d6 "
"#c81a21 "
"#ff6f2f "
"#5e6b4a "
"#dab8cb "
"#ffcdc7 "
"#d58ad3 "
"#633a40 "
"#2b3d7f "
"#ff7ddc "
"#868299 "
"#9bdab6 "
"#d4a1fd "
"#a4411e "
"#c995a3 "
"#e6e1fe "
"#7e65dc "
"#fed3a3 "
"#501a7c "
"#be608a "
"#b45b1b "
"#7e4a40 "
"#84d9ff "
"#495e55 "
"#33aebf "
"#d2e654 "
"#82b5c4 "
"#3f75a1 "
"#80825b "
"#cbad8e "
"#e88246 "
"#cd9879 "
"#2a4b00 "
"#629fbd "
"#73677e "
"#0540c3 "
"#7f00da "
"#ff5be1 "
"#4e6100 "
"#c975ca "
"#7c5287 "
"#6dc5b2 "
"#ffb000 "
"#006b63 "
"#bba561 "
"#701d1a "
"#3d6683 "
"#aab2c4 "
"#929488 "
"#987e65 "
"#dbc1fb "
"#e449a9 "
"#9faaa7 "
"#7979db "
"#785f19 "
"#7fa1a3 "
"#008c71 "
"#56ffc7 "
"#ea099c "
"#9de386 "
"#26ff47 "
"#8d3a7c "
"#5fa6fd "
"#00cc64 "
"#76b47a "
"#607b5a "
"#5738b8 "
"#5f1d4e "
"#679650 "
"#6a8100 "
"#e5ffa9 "
"#7579b8 "
"#bbacc4 "
"#018e91 "
"#006715 "
"#5633ff "
"#beccff "
"#8f92f1 "
"#c4d14a "
"#75006c "
"#adff00 "
"#974b76 "
"#b17abd "
"#67685f "
"#c6c1d2 "
"#525968 "
"#0ad8a8 "
"#c9696f "
"#554b74 "
"#9c1cff "
"#ffc5ff "
"#504e42 "
"#a06772 "
"#e7dee1 "
"#5f3a6f "
"#701d45 "
"#b7b5b0 "
"#c98878 "
"#56df55 "
"#5662a6 "
"#cd8190 "
"#f38f78 "
"#c8d180 "
"#642c00 "
"#5e5dde "
"#8693ca "
"#6caa85 "
"#88003f "
"#7b40a8 "
"#8cc59b "
"#9d1e8b "
"#ff6bad "
"#eb54e7 "
"#78e72c "
"#cb05b3 "
"#afbbc0 "
"#e2a9c9 "
"#a5907e "
"#92fbba "
"#275a92 "
"#b56c39 "
"#b78535 "
"#8d5931 "
"#849666 "
"#035b6f "
"#586688 "
"#4e4e25 "
"#d0284f "
"#794353 "
"#5ac200 "
"#00bae8 "
"#eb8496 "
"#acabe6 "
"#472685 "
"#ecaba1 "
"#ab0080 "
"#c1ace0 "
"#324d5a "
"#91a63d "
"#d49627 "
"#56a62e "
"#5e6f73 "
"#008949 "
"#ffe857 "
"#e6d67c "
"#127385 "
"#15a0dc "
"#466bff "
"#005b45 "
"#e89c62 "
"#768298 "
"#81a6bd "
"#bca599 "
"#c57c5b "
"#486b64 "
"#955553 "
"#5f3c1f "
"#b864bc "
"#f8eb9c "
"#26d6e5 "
"#5883bc "
"#67533d "
"#6e266e "
"#ff00d7 "
"#4225b8 "
"#747e44 "
"#abb136 "
"#a081d0 "
"#b000ae "
"#882a67 "
"#84816e "
"#614963 "
"#cbe5b5 "
"#7aac5d "
"#6d5877 "
"#c2b45d "
"#00bc9e "
"#72eba6 "
"#ec8c07 "
"#736646 "
"#be3bb8 "
"#8f3a58 "
"#89132d "
"#ca3801 "
"#749187 "
"#6fecd8 "
"#636474 "
"#ff4c31 "
"#701faf "
"#91adfc "
"#5eb9c2 "
"#99271d "
"#9a46db "
"#d0e6ee "
"#909a9f "
"#02f38c "
"#404fc2 "
"#a4c857 "
"#7aedff "
"#4c5250 "
"#ed7eff "
"#ffb686 "
"#8f7a4b "
"#5c4649 "
"#479273 "
"#8a86b9 "
"#7f7128 "
"#bc8dc9 "
"#447d70 "
"#4a8053 "
"#6d8ca8 "
"#b0469b "
"#afcdbd "
"#8c688c "
"#bd3438 "
"#758183 "
"#00a358 "
"#f0d7eb "
"#a485a7 "
"#635589 "
"#3e6fb3 "
"#70b8e9 "
"#e5f0b9 "
"#edfff9 "
"#99ad9a "
"#73b0af "
"#5b2f8d "
"#03622d "
"#83acdf "
"#3f5213 "
"#b65d6a "
"#dcffe1 "
"#d8b36c "
"#61d26c "
"#9e1f31 "
"#cca6ad "
"#9cbbb9 "
"#620082 "
"#7e707b "
"#a18435 "
"#c967b6 "
"#d58cff "
"#a96584 "
"#7ed2c8 "
"#8c3200 "
"#214b1d "
"#84be6e "
"#2f2e94 "
"#7acb89 "
"#4ca98b "
"#f57267 "
"#ab8ee8 "
"#ff6747 "
"#884922 "
"#d5588e "
"#a01f52 "
"#40d98a "
"#cc6f31 "
"#bfc022 "
"#454689 "
"#407922 "
"#7e8d4b "
"#a3501e "
"#e13a22 "
"#94c9e9 "
"#20019f "
"#a00e00 "
"#f3bcd2 "
"#f44a5e "
"#0965cb "
"#b4a042 "
"#77438c "
"#ac9028 "
"#cca9d0 "
"#2195b6 "
"#a9ebcb "
"#ffe3ca "
"#fff2f5 "
"#9483a8 "
"#cedcd9 "
"#876b65 "
"#c3791c "
"#76a397 "
"#a24d43 "
"#796c92 "
"#5f6833 "
"#8e82e1 "
"#99fff5 "
"#896ab1 "
"#4c933c "
"#31693c "
"#6a896e "
"#8d5c03 "
"#56af69 "
"#48829f "
"#802c3a "
"#e85f85 "
"#cd51ec "
"#9fb38c "
"#51604a "
"#b7316c "
"#00f1f3 "
"#893b37 "
"#671423 "
"#a39db5 "
"#e5e87f "
"#056e57 "
"#d6aeee "
"#9806d1 "
"#3f5a80 "
"#191bca "
"#9f9cd0 "
"#f67100 "
"#c75839 "
"#e563ba "
"#3c108e "
"#4ea9aa "
"#c1ff7f "
"#efd1d6 "
"#b09464 "
"#78564a "
"#967e93 "
"#ad79a4 "
"#b230fe "
"#734c33 "
"#b230d4 "
"#38783d "
"#f3f5ff "
"#677a6c "
"#a367fe "
"#bca3ff "
"#17ac98 "
"#ef7ea8 "
"#c34e92 "
"#69dbf3 "
"#94d69d "
"#cffff7 "
"#79ff7a "
"#9f48b5 "
"#4539c7 "
"#a8e0ff "
"#b07a8f "
"#539aa4 "
"#701bde "
"#b0a0b0 "
"#a94474 "
"#aa867f "
"#95cdd9 "
"#6b90c4 "
"#e85a00 "
"#8e7711 "
"#b6a9ac "
"#315853 "
"#fea0ff "
"#cac5e6 "
"#e7ad74 "
"#7a6469 "
"#c5d9eb "
"#5e8247 "
"#31416a "
"#86da49 "
"#849e16 "
"#a3b0d8 "
"#b6624d "
"#6b3c6a "
"#4c4aa8 "
"#637ee1 "
"#e371e6 "
"#8c308f "
"#eec76d "
"#5d5b62 "
"#a6f0ff "
"#933fbf "
"#83a48d "
"#685550 "
"#fffed3 "
"#712006 "
"#b54843 "
"#714c51 "
"#5a205b "
"#813c06 "
"#75003b "
"#8b8200 "
"#00ffe6 "
"#c88cad "
"#004975 "
"#97beff "
"#a4bbd5 "
"#ff5d93 "
"#529ce0 "
"#595c55 "
"#786f5f "
"#745524 "
"#af7679 "
"#5fea77 "
"#7e9336 "
"#a58c98 "
"#ffac40 "
"#999597 "
"#916845 "
"#6f2982 "
"#015d59 "
"#9400b5 "
"#5c3c54 "
"#a997c5 "
"#cabbc5 "
"#eecbff "
"#cb4878 "
"#842b14 "
"#9c6805 "
"#b9c0a0 "
"#d5a10a "
"#4d5a8e "
"#6b6e75 "
"#646199 "
"#e039e6 "
"#99dbdc "
"#d60986 "
"#b27b53 "
"#45c7a1 "
"#ffaf96 "
"#651e34 "
"#8d5d6b "
"#59bdb6 "
"#78c353 "
"#ae0239 "
"#6f0356 "
"#7e7d7e "
"#786900 "
"#925eea "
"#b9e540 "
"#be92b9 "
"#51543a "
"#604aab "
"#828c71 "
"#547190 "
"#c9dcca "
"#efc749 "
"#f39246 "
"#b265e5 "
"#bbb6d4 "
"#d593e8 "
"#c29742 "
"#96519d "
"#7a356a "
"#d28851 "
"#a25695 "
"#9f9947 "
"#002fff "
"#5ab423 "
"#891517 "
"#88a57a "
"#9e4487 "
"#dfd72e "
"#00e2c7 "
"#356f6e "
"#008acb "
"#725a8a "
"#7790ea "
"#d635a2 "
"#b03800 "
"#c297e7 "
"#5b5217 "
"#669772 "
"#ce6e5f "
"#f03465 "
"#ff9fdd "
"#e9001b "
"#6e5368 "
"#ab34b4 "
"#c8c4bc "
"#bdc8d3 "
"#fff440 "
"#d23640 "
"#5a6edd "
"#7b4cba "
"#b1691b "
"#78748b "
"#08847c "
"#969dae "
"#8f8d5b "
"#ca76ae "
"#6645f2 "
"#ea84e3 "
"#ad76d3 "
"#624300 "
"#5a7e27 "
"#489a8a "
"#935002 "
"#cb0056 "
"#cd515f "
"#770002 "
"#3b7a84 "
"#86458b "
"#81b9a0 "
"#f374c3 "
"#b18006 "
"#4e87d9 "
"#d0ff00 "
"#27667a "
"#ffb3e1 "
"#ffe3a9 "
"#6a3319 "
"#e1bbeb "
"#027ef7 "
"#b6c8c6 "
"#8c8a95 "
"#bfaa7b "
"#a35cb8 "
"#c7ffe5 "
"#e089c7 "
"#cae589 "
"#85b034 "
"#d7794c "
"#e7e0c9 "
"#ff42a7 "
"#254ba2 "
"#ddc8a2 "
"#e3566c "
"#358928 "
"#58766e "
"#8a96b3 "
"#ca6a87 "
"#5722a3 "
"#6177b7 "
"#99baa8 "
"#056391 "
"#fbbbb2 "
"#a77b3a "
"#640670 "
"#ba3259 "
"#917779 "
"#c42001 "
"#a09a7f "
"#92005e "
"#00f05a "
"#bdad3c "
"#9775e2 "
"#810966 "
"#d69bb6 "
"#d5cfb9 "
"#5a8e7e "
"#9a715a "
"#ea3d8f "
"#cfcb61 "
"#db693f "
"#cb9290 "
"#e18373 "
"#e2a937 "
"#81569f "
"#9e5d49 "
"#7f8430 "
"#8b8681 "
"#59483e "
"#a76cae "
"#6f7a9f "
"#379600 "
"#217b9a "
"#3d514b "
"#ff418e "
"#7e65ff "
"#d77d34 "
"#8b4f5b "
"#335036 "
"#422e96 "
"#f39296 "
"#423270 "
"#4ee7b8 "
"#44c5e7 "
"#89759e "
"#51b37f "
"#c8d89d "
"#7b34ff "
"#339554 "
"#b56f97 "
"#eeff8c "
"#416c00 "
"#d267e2 "
"#9ece00 "
"#622485 "
"#ffcaed "
"#95b1c4 "
"#ee7446 "
"#004f36 "
"#d6d6d9 "
"#8cb0b4 "
"#bb80ff "
"#665f5e "
"#00aa46 "
"#9d692a "
"#9dde66 "
"#2c32b6 "
"#ffe3ef "
"#556922 "
"#6a6e00 "
"#e0987f "
"#da6b7e "
"#e96aaa "
"#9c8e00 "
"#8930af "
"#da3cbe "
"#605677 "
"#476634 "
"#8386cc "
"#5002fe "
"#ccffa8 "
"#a7acb2 "
"#736e6d "
"#9a9ce5 "
"#a5cc76 "
"#4f5f66 "
"#845377 "
"#a36692 "
"#f5d73b "
"#637c7f "
"#500095 "
"#875843 "
"#4e495b "
"#c24823 "
"#9cf57f "
"#5dc8d5 "
"#734a00 "
"#3c4e6b "
"#e5c6de "
"#5bc487 "
"#9fed4e "
"#bfbe79 "
"#429594 "
"#b959aa "
"#e4e4f0 "
"#6e5cce "
"#3661aa "
"#b8b1ff "
"#b9cfee "
"#af2475 "
"#826e3f "
"#ec00d6 "
"#d66400 "
"#5e4f26 "
"#4c6a50 "
"#25526a "
"#730024 "
"#a23e4a "
"#ce3422 "
"#3f82b3 "
"#96f0d8 "
"#e5a6f2 "
"#bcf18c "
"#612d50 "
"#05a173 "
"#5c8d8a "
"#81644f "
"#8e7085 "
"#695c6b "
"#dc929e "
"#44515d "
"#58db27 "
"#cb8b33 "
"#a7c08c "
"#e7e1a9 "
"#ff864c "
"#d656b4 "
"#8f0001 "
"#c80ce7 "
"#b07b68 "
"#e6f2e5 "
"#ebd8d0 "
"#fd9f80 "
"#ee0249 "
"#819292 "
"#f5e1ff "
"#dab0ad "
"#d24165 "
"#743e31 "
"#926565 "
"#77add1 "
"#3aab00 "
"#678a99 "
"#e2ff46 "
"#8fba85 "
"#67e4ea "
"#005a00 "
"#8745dd "
"#91cb43 "
"#7a8892 "
"#648aff "
"#77c8a6 "
"#00c33a "
"#00978a "
"#936ba6 "
"#a99692 "
"#b5e4b3 "
"#4a4d7b "
"#e9ed64 "
"#3b8861 "
"#9b2f62 "
"#6a010c "
"#667547 "
"#58411a "
"#9772c6 "
"#5c5f3a "
"#33684c "
"#318eff "
"#4fc056 "
"#fff4da "
"#e99a35 "
"#595cbd "
"#784a1e "
"#ffc3cc "
"#b7cc15 "
"#ad2610 "
"#4fd3c5 "
"#a191b1 "
"#459fb6 "
"#4f6f7a "
"#d8512c "
"#eeb566 "
"#ffe57e "
"#b95f36 "
"#445636 "
"#7f5daf "
"#958077 "
"#add8e1 "
"#6f2d3c "
"#bd0588 "
"#bff015 "
"#9b8cc1 "
"#cdc3ff "
"#d9b453 "
"#98a195 "
"#90d482 "
"#3562e3 "
"#ecc6b8 "
"#ccdd6a "
"#0173a9 "
"#dfaf8b "
"#ea842a "
"#4a4a0f "
"#ac8fd2 "
"#5270a2 "
"#afa600 "
"#a7d0a3 "
"#aca9b9 "
"#5cb1f5 "
"#2f6262 "
"#7c5b69 "
"#7f67c1 "
"#8689a9 "
"#624730 "
"#7b6185 "
"#786232 "
"#98b2e9 "
"#578aaf "
"#be5f9c "
"#8bd5bd "
"#556cbb "
"#7c2855 "
"#e15346 "
"#f31d39 "
"#9437a4 "
"#5933de "
"#52f303 "
"#ff43cd "
"#ffd2b9 "
"#9ff7a6 "
"#e2d8eb "
"#ef4d1c "
"#9492a7 "
"#9ab934 "
"#c2339b "
"#b66a6b "
"#475e1d "
"#3a5b49 "
"#895c8d "
"#cbc9a0 "
"#7b7551 "
"#5c004a "
"#bdb392 "
"#6e7026 "
"#8e7fff "
"#337164 "
"#65727f "
"#d96eff "
"#acf3c1 "
"#00d757 "
"#84442f "
"#649bc9 "
"#9aa562 "
"#aa53e8 "
"#9b4665 "
"#6c394c "
"#d29b51 "
"#9e7401 "
"#2479d5 "
"#824678 "
"#ff92bd "
"#580001 "
"#ba99ad "
"#64cff9 "
"#30acde "
"#fffef1 "
"#a54300 "
"#d49764 "
"#bc48e6 "
"#564071 "
"#ccefed "
"#ada172 "
"#767a6f "
"#ec99b4 "
"#e4b0c0 "
"#73467a "
"#60120f "
"#378d9d "
"#d7f765 "
"#ed845e "
"#839286 "
"#4e5f77 "
"#d5614b "
"#07f8b1 "
"#50256a "
"#c3767d "
"#c78a65 "
"#f1c989 "
"#00c1b1 "
"#882142 "
"#6e8e20 "
"#bfadba "
"#738266 "
"#317e64 "
"#dc62c8 "
"#879ae3 "
"#004f54 "
"#d9b6d7 "
"#5f1900 "
"#4e50de "
"#6d9b00 "
"#cbd1ed "
"#843c4b "
"#c07bd5 "
"#cfcb8c "
"#b5854e "
"#b45acd "
"#860093 "
"#919825 "
"#267224 "
"#6e68e2 "
"#6d652a "
"#439bc4 "
"#8ec1d5 "
"#5f535d "
"#550061 "
"#754660 "
"#6e989e "
"#6f4241 "
"#eb6354 "
"#635800 "
"#7b37d0 "
"#d2bfd5 "
"#531ebd "
"#a4d5c6 "
"#00ad6e "
"#355c69 "
"#95a1ff "
"#2b5076 "
"#5a9d60 "
"#b25cff "
"#515169 "
"#7b1f29 "
"#73f64e "
"#663fc8 "
"#83e073 "
"#97004f "
"#67380e "
"#eed9ae "
"#45605f "
"#8a9fae "
"#dec2ca "
"#00efdc "
"#c0ecd7 "
"#503e7d "
"#3844fe "
"#be869b "
"#f672e5 "
"#d2af32 "
"#a06b60 "
"#bfcfab "
"#f136ff "
"#3c15a5 "
"#20800e "
"#8d4942 "
"#ffbf36 "
"#7e3718 "
"#c7337c "
"#bddefb "
"#df99e0 "
"#6b3e9d "
"#e5f8d1 "
"#9aa778 "
"#56ff92 "
"#964726 "
"#0d4bd5 "
"#696a53 "
"#8f00ee "
"#ebbf0f "
"#1c259f "
"#c2b3a4 "
"#b6a3cc "
"#89fbd0 "
"#85beb3 "
"#6f6f96 "
"#13a6ae "
"#004763 "
"#efc6a1 "
"#a85579 "
"#c19a6f "
"#89fbff "
"#d27dc3 "
"#324812 "
"#9a8e4a "
"#595c1b "
"#5c3e61 "
"#484263 "
"#9a1e6c "
"#7de1da "
"#921f04 "
"#799f5a "
"#552f62 "
"#567762 "
"#6e7c2c "
"#7d2cbd "
"#0d7246 "
"#94ad59 "
"#ff8a01 "
"#74cc38 "
"#6fad96 "
"#963236 "
"#98697e "
"#78bcdc "
"#ff49ff "
"#39bf8d "
"#0394c7 "
"#ea89b9 "
"#a77f9a "
"#3f3a7e "
"#752d2e "
"#f6a0aa "
"#24477b "
"#8274b1 "
"#f06c9a "
"#d8880e "
"#a7a786 "
"#3f8180 "
"#4e7835 "
"#04b230 "
"#c051bf "
"#6d4a8e "
"#89e1ad "
"#fe8dd5 "
"#bb74b5 "
"#a3aaf7 "
"#2296e1 "
"#817840 "
"#8a4fbd "
"#00b7d4 "
"#a27bbc "
"#3f9e36 "
"#7ca8f2 "
"#8c7565 "
"#b1445e "
"#8abaed "
"#4db154 "
"#a5896a "
"#d10070 "
"#b9de5c "
"#68195f "
"#456374 "
"#598a29 "
"#c74a4a "
"#d28584 "
"#939dc6 "
"#fb3ee2 "
"#a45b65 "
"#5c5fff "
"#8a4761 "
"#2d2284 "
"#eda5d6 "
"#c5f2fa "
"#6b2f60 "
"#ffa775 "
"#925c20 "
"#51575c "
"#366e78 "
"#a7b856 "
"#ff7584 "
"#b14b1f "
"#897c5d "
"#ab93aa "
"#7c5500 "
"#538090 "
"#64cbca "
"#a25335 "
"#296525 "
"#5e397f "
"#b2da00 "
"#b5ce88 "
"#776473 "
"#80464a "
"#e8eb3b "
"#ffc590 "
"#5c4b6c "
"#87a5ca "
"#f1b7f0 "
"#b185b6 "
"#8fa7a4 "
"#e9f19c "
"#ff596f "
"#f36128 "
"#7d53e4 "
"#7926a1 "
"#682d23 "
"#c3a5ee "
"#d273a1 "
"#95f52b "
"#bc0024 "
"#f0caf0 "
"#428873 "
"#a5ffe5 "
"#316a96 "
"#eb738c "
"#5da494 "
"#eb2000 "
"#5e865e "
"#2c5d78 "
"#a497a1 "
"#ff8597 "
"#7f1648 "
"#5e3f9d "
"#afa692 "
"#ebf8fc "
"#c4ffd3 "
"#c39f88 "
"#ca95cb "
"#8e2579 "
"#b1b400 "
"#964235 "
"#007c43 "
"#8f5ca1 "
"#0e4087 "
"#d3b281 "
"#fff2b7 "
"#8db300 "
"#844600 "
"#5c5e78 "
"#d3023c "
"#f0b598 "
"#004c25 "
"#bc477e "
"#ae422f "
"#c19e9c "
"#a75300 "
"#2ec777 "
"#af574e "
"#9c7279 "
"#729561 "
"#3982c9 "
"#cf7c6d "
"#35488d "
"#8f9673 "
"#b31465 "
"#9a9e9d "
"#b9827a "
"#39724d "
"#b0dec4 "
"#376dce "
"#7e76f9 "
"#a43421 "
"#c5cfc9 "
"#c583f2 "
"#a0cdaf "
"#5e564a "
"#a605b7 "
"#b8ca6e "
"#ffa111 "
"#2c59cb "
"#a438d5 "
"#df9f9c "
"#faffa1 "
"#3144ba "
"#e56d6c "
"#cfc7ca "
"#a9a15c "
"#4f9e7a "
"#b898d2 "
"#b5e38f "
"#ffc95b "
"#cec640 "
"#706295 "
"#6869ac "
"#b7bdae "
"#d7ba98 "
"#50b696 "
"#775abc "
"#c6f4bc "
"#ce2eff "
"#ab0013 "
"#c863f3 "
"#66efe9 "
"#bd8f24 "
"#970028 "
"#d5b81a "
"#b296ff "
"#ac3788 "
"#687faf "
"#606c69 "
"#9081c7 "
"#bbc0ff "
"#cd4d00 "
"#255e41 "
"#ff464d "
"#7871c5 "
"#b08f91 "
"#50633e "
"#d9f2ab "
"#6d397b "
"#26af5e "
"#4277e8 "
"#6cf3cf "
"#ec966e "
"#6a412c "
"#be2d47 "
"#c04334 "
"#d5bdb0 "
"#006354 "
"#a762cd "
"#94dc23 "
"#ea8585 "
"#943d6a "
"#968a93 "
"#0970c2 "
"#705e49 "
"#a6ebe4 "
"#e13335 "
"#693434 "
"#659539 "
"#c05c79 "
"#631037 "
"#6fbdd0 "
"#81a7b2 "
"#480076 "
"#8f6ad2 "
"#886377 "
"#2a5514 "
"#b8bcc9 "
"#53e691 "
"#97b96b "
"#b134a6 "
"#ced9e0 "
"#d184b5 "
"#92822b "
"#b5762f "
"#a90026 "
"#c24c67 "
"#afc5d7 "
"#c4b270 "
"#5b5b87 "
"#edf1d4 "
"#9ec8fb "
"#315052 "
"#dba279 "
"#ee67ff "
"#8772e0 "
"#667cff "
"#c35b00 "
"#a5b16e "
"#746f7c "
"#a678ff "
"#a093ef "
"#843a96 "
"#addad6 "
"#f2a04f "
"#f058af "
"#835f57 "
"#b7ba8b "
"#ba82a7 "
"#cabaea "
"#4fcf9b "
"#d0afa0 "
"#806982 "
"#493ca1 "
"#0e4e00 "
"#78bf8b "
"#d8dff8 "
"#a082e6 "
"#82949d "
"#5d666e "
"#7c2684 "
"#4da9c7 "
"#019624 "
"#948b7b "
"#5b8b41 "
"#826540 "
"#f93628 "
"#64316a "
"#6b6a83 "
"#8dd2ea "
"#6d2850 "
"#f0a2ea "
"#ee34bc "
"#bea310 "
"#acc9ce "
"#945120 "
"#76a07d "
"#4a79b2 "
"#519e00 "
"#aec03e "
"#92703c "
"#d3cce2 "
"#dac387 "
"#aba9a9 "
"#00dd3d "
"#9f9061 "
"#755a59 "
"#6cb662 "
"#b2bade "
"#b045c0 "
"#82ffa7 "
"#cda75c "
"#3aa7ec "
"#ef94ef "
"#7aca74 "
"#6f18c5 "
"#f0297e "
"#efebef "
"#933014 "
"#a9b1a7 "
"#749995 "
"#ff85a9 "
"#a3a7bf "
"#40476b "
"#d07317 "
"#446757 "
"#7f86e2 "
"#93c565 "
"#ac00dc "
"#d7db7c "
"#5edf9f "
"#675e75 "
"#415801 "
"#8d00d7 "
"#7c1e13 "
"#2b6600 "
"#a960a5 "
"#bd7058 "
"#815031 "
"#6a78cb "
"#634f98 "
"#de266c "
"#e1ca63 "
"#66a84c "
"#89e999 "
"#efa08e "
"#e37000 "
"#253e97 "
"#74505f "
"#a7c7e5 "
"#7b8414 "
"#5d008e "
"#e1e6bf "
"#c988d8 "
"#a8b9a4 "
"#008ba7 "
"#003484 "
"#de5d27 "
"#c3e0df "
"#ca00c7 "
"#2c7972 "
"#95c3c7 "
"#5f5638 "
"#807988 "
"#e6d7ff "
"#6601ea "
"#66bf39 "
"#785579 "
"#7d8da7 "
"#632da7 "
"#9e674a "
"#621d43 "
"#a12143 "
"#8b5aff "
"#627890 "
"#3d8951 "
"#fff76a "
"#bf5421 "
"#676768 "
"#cab2c9 "
"#00e3ea "
"#e3cc3f "
"#adc4ff "
"#d37f9e "
"#ee5d63 "
"#a2b5c0 "
"#ed364d "
"#b06dc3 "
"#cfe7fe "
"#3409ff "
"#efffc7 "
"#c4b93a "
"#493a68 "
"#adb7eb "
"#9b3905 "
"#70d357 "
"#7c0298 "
"#9d5684 "
"#a27a58 "
"#e38e58 "
"#ac5f75 "
"#80e4c8 "
"#345731 "
"#cccc1a "
"#9746ef "
"#f134a3 "
"#b8909d "
"#6c777c "
"#d12fde "
"#2ec1c7 "
"#826f72 "
"#d3539e "
"#d9e4d9 "
"#5356a7 "
"#f37931 "
"#49b9f0 "
"#fd005e "
"#6c502e "
"#f6816e "
"#a76f82 "
"#a1c5bb "
"#ed4d86 "
"#d5b4bc "
"#687164 "
"#8baf9b "
"#429769 "
"#b89479 "
"#bd76e5 "
"#fc6401 "
"#89a267 "
"#a1b2b4 "
"#687a76 "
"#6eaac1 "
"#7cdcb5 "
"#694360 "
"#bfc6ec "
"#7d266f "
"#60f468 "
"#002191 "
"#6c927c "
"#94855b "
"#ff6686 "
"#905572 "
"#c11a64 "
"#b9515b "
"#a86227 "
"#df4ad2 "
"#b13c45 "
"#62a873 "
"#896b24 "
"#6d49aa "
"#659aac "
"#b8bdbc "
"#94959f "
"#9a4f5a "
"#544a83 "
"#d76fba "
"#68a660 "
"#613f3a "
"#5dce7e "
"#907985 "
"#835a28 "
"#387191 "
"#c1bcbe "
"#6b92d9 "
"#d3a3c7 "
"#cb308f "
"#733605 "
"#541965 "
"#6d5b39 "
"#ba6cff "
"#70c0ff "
"#e13afe "
"#d6ddc6 "
"#f35445 "
"#556e39 "
"#bf6624 "
"#3004a0 "
"#b0e4f2 "
"#d74f51 "
"#1d8288 "
"#d80019 "
"#57635c "
"#be3f6a "
"#f3ba7e "
"#80c5d0 "
"#80848a "
"#48b640 "
"#a74e67 "
"#5a8d6c "
"#5df6ff "
"#aef968 "
"#bcb0ed "
"#008761 "
"#957393 "
"#60725a "
"#c09ec9 "
"#ade6d5 "
"#7a7c98 "
"#ecaeb3 "
"#87335d "
"#189674 "
"#462672 "
"#3db1a7 "
"#18584d "
"#b29a8a "
"#849dce "
"#6a653b "
"#6d8c4c "
"#c27446 "
"#bf798d "
"#ff6ad3 "
"#8dcecf "
"#eedc93 "
"#d4f1ff "
"#c45cd4 "
"#4e6c6e "
"#59794a "
"#ffb95a "
"#84e3ff "
"#ff9d58 "
"#4dbfa9 "
"#998145 "
"#fab0ff "
"#f2e169 "
"#dc7b8e "
"#ac2953 "
"#868a7d "
"#a16e9e "
"#ddf0ee "
"#c39d5f "
"#608299 "
"#f0c6d2 "
"#5d6656 "
"#7732e5 "
"#aba1eb "
"#76f5ba "
"#6494b6 "
"#71aa35 "
"#5c4900 "
"#cfccd5 "
"#76c4f0 "
"#eecc00 "
"#b4ae66 "
"#578281 "
"#e7b939 "
"#717b55 "
"#dee000 "
"#11da99 "
"#af608f "
"#04785a "
"#ff96ef "
"#ccef43 "
"#ff696e "
"#465222 "
"#955d8c "
"#715e00 "
"#603b4b "
"#005fff "
"#5d9098 "
"#793b58 "
"#485369 "
"#457b5b "
"#af9400 "
"#a0a341 "
"#9ab2ab "
"#7e69a0 "
"#4f5b5a "
"#95008e "
"#b6fff5 "
"#d72d7b "
"#c7e4c5 "
"#006382 "
"#800430 "
"#e0a994 "
"#5e8900 "
"#8ac602 "
"#005da7 "
"#72b600 "
"#7af8e9 "
};

/********************************************/
#include "pbardefs_float.h"  /* 24 Oct 2019 */
/********************************************/

#endif /* __PBARDEFS_INCLUDED__ */
